// ===================================================================
// CONFIGURATION - Error Logging
// ===================================================================
const SHOW_ERRORS = false; // Set to true to show errors in console

// Override console.error and console.warn to suppress messages
if (!SHOW_ERRORS) {
  const originalError = console.error;
  const originalWarn = console.warn;

  console.error = function (...args) {
    // Suppress all error logs in production
    return;
  };

  console.warn = function (...args) {
    // Suppress all warning logs in production
    return;
  };
}

// ===================================================================
// GLOBAL ERROR HANDLERS - Prevent errors from showing in Chrome Extensions
// ===================================================================

// Catch unhandled promise rejections
self.addEventListener('unhandledrejection', (event) => {
  if (!SHOW_ERRORS) {
    event.preventDefault(); // Prevent the error from appearing in Chrome's error log
  }
});

// Catch runtime errors
self.addEventListener('error', (event) => {
  if (!SHOW_ERRORS) {
    event.preventDefault(); // Prevent the error from appearing in Chrome's error log
  }
});

// ===================================================================
// STATE MANAGEMENT
// ===================================================================

// Note: escapeHtml() is available globally from security-helpers.js

// State management
let state = {
  isLoggedIn: false,
  userEmail: '',
  userPlan: 'Free',
  fiverrUsername: '',
  timeInterval: null, // ✨ Changed from 30 to null (User must set it)
  botStatus: 'Offline',
  backendUrl: 'https://fivreply-production.up.railway.app',
  gigs: [],
  refreshMode: false,  // ✨ NEW: Refresh mode state
  subscription: {  // ✨ NEW: Subscription info
    status: 'trial',
    startDate: null,
    endDate: null
  },
  analytics: {
    totalReplies: 0,
    repliesToday: 0,  // ✅ Today's count only
    avgTime: 0,
    successRate: 0,
    activeChats: 0,  // ✅ Lifetime unique customers (Analytics tab)
    activeConversations: 0  // ✅ Today's unique customers (Bot Controls)
  },
  logs: []
};

// Toast Notification System
function showToast(title, message, type = 'info', duration = 4000) {
  const container = document.getElementById('toastContainer');
  if (!container) return;

  // Use secure helper function to build toast (prevents XSS)
  const toast = window.SecurityHelpers.buildToastElement(title, message, type);

  container.appendChild(toast);

  // Close button
  const closeBtn = toast.querySelector('.toast-close');
  closeBtn.addEventListener('click', () => removeToast(toast));

  // Auto remove
  setTimeout(() => removeToast(toast), duration);
}

function removeToast(toast) {
  toast.classList.add('hiding');
  setTimeout(() => {
    if (toast.parentNode) {
      toast.parentNode.removeChild(toast);
    }
  }, 300);
}

// ⚠️ NOTE: These are FALLBACK limits only!
// Real limits come from backend database
const planLimits = {
  Free: { gigs: 2, repliesPerMonth: 20 },  // Updated to match database
  Basic: { gigs: 5, repliesPerMonth: 100 },
  Professional: { gigs: 10, repliesPerMonth: 300 },
  Enterprise: { gigs: 20, repliesPerMonth: 1000 },
  'Yearly Professional': { gigs: 10, repliesPerMonth: -1 },
  Lifetime: { gigs: -1, repliesPerMonth: -1 }
};

const DEFAULT_PLAN_NAME = 'Free';

function getPlanKey(planName = DEFAULT_PLAN_NAME) {
  if (!planName || typeof planName !== 'string') return DEFAULT_PLAN_NAME;
  const normalized = planName.trim().toLowerCase();
  const match = Object.keys(planLimits).find(key => key.toLowerCase() === normalized);
  return match || DEFAULT_PLAN_NAME;
}

function getPlanLimitsFor(planName = DEFAULT_PLAN_NAME) {
  return planLimits[getPlanKey(planName)] || planLimits[DEFAULT_PLAN_NAME];
}

function normalizePlanLimits(planName = DEFAULT_PLAN_NAME, limits = {}) {
  const fallback = getPlanLimitsFor(planName);

  const gigs =
    typeof limits.gigs === 'number'
      ? limits.gigs
      : typeof limits.maxGigs === 'number'
        ? limits.maxGigs
        : fallback.gigs;

  const repliesPerMonth =
    typeof limits.repliesPerMonth === 'number'
      ? limits.repliesPerMonth
      : typeof limits.maxRepliesPerMonth === 'number'
        ? limits.maxRepliesPerMonth
        : typeof limits.replies === 'number'
          ? limits.replies
          : fallback.repliesPerMonth;

  return {
    gigs,
    repliesPerMonth
  };
}

function getEffectivePlanLimits() {
  return normalizePlanLimits(state.userPlan, state.planLimits);
}

// Check if subscription has expired
function isPlanExpired() {
  if (!state.subscription || !state.subscription.endDate) {
    // No end date means Lifetime plan or not set - not expired
    return false;
  }

  const endDate = new Date(state.subscription.endDate);
  const now = new Date();

  return endDate < now;
}

state.planLimits = getPlanLimitsFor(state.userPlan);

// ========================================
// ANNOUNCEMENT BAR FUNCTIONS
// ========================================

/**
 * Show announcement bar with custom message and optional link
 * @param {string} message - The announcement message
 * @param {string} linkText - Optional link text
 * @param {string} linkUrl - Optional link URL
 * @param {string} theme - Color theme: 'yellow' (default), 'blue', 'green', 'red'
 */
function showAnnouncement(message, linkText = '', linkUrl = '', theme = 'yellow') {
  const announcementBar = document.getElementById('announcementBar');
  const announcementMessage = document.getElementById('announcementMessage');
  const announcementLink = document.getElementById('announcementLink');

  if (!announcementBar) return;

  // Set message
  announcementMessage.textContent = message;

  // Set link if provided
  if (linkText && linkUrl) {
    announcementLink.textContent = linkText;
    announcementLink.href = linkUrl;
    announcementLink.classList.remove('hidden');
  } else {
    announcementLink.classList.add('hidden');
  }

  // Apply theme
  announcementBar.className = 'announcement-bar';
  if (theme && theme !== 'yellow') {
    announcementBar.classList.add(theme);
  }

  // Show the announcement bar
  announcementBar.classList.remove('hidden');

  // Store that we've shown this announcement
  const announcementKey = `announcement_${Date.now()}`;
  localStorage.setItem('lastAnnouncement', announcementKey);
}

/**
 * Hide announcement bar
 */
function hideAnnouncement() {
  const announcementBar = document.getElementById('announcementBar');
  if (announcementBar) {
    announcementBar.classList.add('hidden');
  }
}

/**
 * Load announcement from backend API
 */
async function loadAnnouncement() {
  try {
    const response = await fetch(`${state.backendUrl}/api/announcement`);

    if (response.ok) {
      const data = await response.json();

      // Check if announcement is active
      if (data.active && data.message) {
        // Check if user has dismissed this specific announcement
        const dismissedAnnouncements = JSON.parse(localStorage.getItem('dismissedAnnouncements') || '[]');

        if (!dismissedAnnouncements.includes(data.id)) {
          showAnnouncement(
            data.message,
            data.linkText || '',
            data.linkUrl || '',
            data.theme || 'yellow'
          );
        }
      }
    }
  } catch (error) {
    console.log('Could not load announcement:', error.message);
    // Silently fail - announcement is optional
  }
}

/**
 * Dismiss announcement and remember it
 */
function dismissAnnouncement(announcementId = null) {
  hideAnnouncement();

  if (announcementId) {
    // Store dismissed announcement ID
    const dismissedAnnouncements = JSON.parse(localStorage.getItem('dismissedAnnouncements') || '[]');
    if (!dismissedAnnouncements.includes(announcementId)) {
      dismissedAnnouncements.push(announcementId);
      localStorage.setItem('dismissedAnnouncements', JSON.stringify(dismissedAnnouncements));
    }
  }
}

// ========================================
// PLANS & BILLING FUNCTIONS
// ========================================

/**
 * Load plans from API and display in UI
 */
async function loadPlans() {
  console.log('📋 Loading plans from API...');

  const plansLoading = document.getElementById('plansLoading');
  const plansGrid = document.getElementById('plansGrid');
  const noPlansMessage = document.getElementById('noPlansMessage');

  try {
    // Show loading state
    plansLoading.style.display = 'flex';
    plansGrid.style.display = 'none';
    noPlansMessage.style.display = 'none';

    // Fetch plans from API
    const result = await api.getPlans();

    if (result.success && result.plans && result.plans.length > 0) {
      console.log(`✅ Loaded ${result.plans.length} plan(s)`);

      // Hide loading, show grid
      plansLoading.style.display = 'none';
      plansGrid.style.display = 'grid';

      // Render plans
      renderPlans(result.plans);
    } else {
      // No plans available
      console.warn('⚠️ No plans available');
      plansLoading.style.display = 'none';
      noPlansMessage.style.display = 'block';
    }
  } catch (error) {
    console.error('❌ Error loading plans:', error);
    plansLoading.style.display = 'none';
    noPlansMessage.style.display = 'block';
    noPlansMessage.innerHTML = `
      <p style="color: #ef4444; margin-bottom: 8px;">Failed to load plans</p>
      <p style="font-size: 12px;">${error.message}</p>
      <button onclick="loadPlans()" class="btn-primary" style="margin-top: 16px;">Retry</button>
    `;
  }
}

/**
 * Render plans in the UI
 */
function renderPlans(plans) {
  const plansGrid = document.getElementById('plansGrid');
  if (!plansGrid) return;

  const currentPlanSlug = (state.userPlan || DEFAULT_PLAN_NAME).toString().trim().toLowerCase();

  plansGrid.innerHTML = plans.map(plan => {
    const isPopular = plan.isPopular || plan.badge;
    const limitText = getLimitText(plan.limits);
    const planSlug = (plan.name || plan.displayName || '').toString().trim().toLowerCase();
    const planLabel = escapeHtml(plan.displayName || plan.name || 'Plan');
    const isCurrent = planSlug === currentPlanSlug;

    return `
      <div class="plan-card ${isPopular ? 'popular' : ''} ${isCurrent ? 'plan-current' : ''}"
           data-plan-id="${escapeHtml(plan._id)}"
           data-plan-name="${planLabel}"
           data-plan-slug="${escapeHtml(plan.name || planLabel)}">
        ${plan.badge ? `<div class="plan-badge">${escapeHtml(plan.badge)}</div>` : ''}
        
        <div class="plan-name">${planLabel}</div>
        
        <div class="plan-price">
          <span class="currency">${getCurrencySymbol(plan.currency)}</span>${escapeHtml(String(plan.price))}
        </div>
        
        <div class="plan-interval">${escapeHtml(getIntervalText(plan.interval, plan.trialDays))}</div>
        
        ${plan.description ? `<div class="plan-description">${escapeHtml(plan.description)}</div>` : ''}
        
        <div class="plan-limits">
          <div class="plan-limits-title">Limits:</div>
          ${limitText}
        </div>
        
        <ul class="plan-features">
          ${Array.isArray(plan.features) ? plan.features.slice(0, 5).map(feature => `
            <li class="${feature.included ? '' : 'not-included'}">
              ${escapeHtml(feature.name)}
            </li>
          `).join('') : ''}
          ${plan.features && plan.features.length > 5 ? `<li style="color: rgba(255,255,255,0.5);">+ ${plan.features.length - 5} more...</li>` : ''}
        </ul>
        
        <button class="btn-plan"
                data-plan-id="${escapeHtml(plan._id)}"
                data-plan-name="${planLabel}">
          Select Plan
        </button>
      </div>
    `;
  }).join('');

  plansGrid.querySelectorAll('.btn-plan').forEach(btn => {
    const planId = btn.getAttribute('data-plan-id');
    const planName = btn.getAttribute('data-plan-name');
    btn.addEventListener('click', () => selectPlan(planId, planName));
  });
}

/**
 * Get currency symbol
 */
function getCurrencySymbol(currency) {
  const symbols = {
    'USD': '$',
    'EUR': '€',
    'GBP': '£',
    'INR': '₹',
    'BDT': '৳'
  };
  return symbols[currency] || '$';
}

/**
 * Get interval text (e.g., "per monthly", "7 days free trial")
 */
function getIntervalText(interval, trialDays) {
  if (interval === 'trial' && trialDays > 0) {
    return `${trialDays} days free trial`;
  }
  if (interval === 'lifetime') {
    return 'one-time payment';
  }
  return `per ${interval}`;
}

/**
 * Get limit text for display
 */
function getLimitText(limits) {
  const items = [];

  if (limits.maxRepliesPerMonth !== undefined) {
    const repliesText = limits.maxRepliesPerMonth === -1
      ? 'Unlimited'
      : limits.maxRepliesPerMonth;
    items.push(`
      <div class="plan-limits-item">
        <span>Replies/Month:</span>
        <span class="plan-limits-value">${repliesText}</span>
      </div>
    `);
  }

  if (limits.maxGigs !== undefined) {
    const gigsText = limits.maxGigs === -1
      ? 'Unlimited'
      : limits.maxGigs;
    items.push(`
      <div class="plan-limits-item">
        <span>Gigs:</span>
        <span class="plan-limits-value">${gigsText}</span>
      </div>
    `);
  }

  if (limits.maxDevices !== undefined) {
    items.push(`
      <div class="plan-limits-item">
        <span>Devices:</span>
        <span class="plan-limits-value">${limits.maxDevices}</span>
      </div>
    `);
  }

  return items.join('');
}

/**
 * Handle plan selection
 */
function selectPlan(planId, planName) {
  console.log(`📋 Plan selected: ${planName} (${planId})`);

  if (!state.isLoggedIn || !api.token) {
    showToast('Authentication Required', 'Please login first to purchase a plan', 'warning', 4000);
    addLog('ERROR', 'Must be logged in to purchase plans');
    showLoginModal();
    return;
  }

  // Show payment gateway selection
  showPaymentGatewayModal(planId, planName);
}

/**
 * Show payment gateway selection modal
 */
async function showPaymentGatewayModal(planId, planName) {
  try {
    // Fetch enabled payment gateways from backend
    const response = await fetch(`${state.backendUrl}/api/admin/payment-gateways`);
    const data = await response.json();

    const enabledGateways = data.success ? data.paymentGateways : {};

    // If no gateways are enabled, show error
    if (Object.keys(enabledGateways).length === 0) {
      showToast('No payment methods available. Please contact support.', 'error');
      return;
    }
    const modal = document.createElement('div');
    modal.id = 'paymentGatewayModal';
    modal.style.cssText = `
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0,0,0,0.8);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10000;
    padding: 20px;
  `;

    // Gateway color schemes
    const gatewayColors = {
      razorpay: '#528FF0',
      stripe: '#635BFF',
      paypal: '#0070BA'
    };

    // Build buttons HTML for enabled gateways
    let buttonsHTML = '';

    if (enabledGateways.razorpay) {
      buttonsHTML += `
        <button class="payment-btn" data-gateway="razorpay"
                style="padding: 16px; background: ${gatewayColors.razorpay}; color: white; border: none; border-radius: 8px; cursor: pointer; font-size: 14px; font-weight: 600; display: flex; align-items: center; justify-content: center; gap: 8px;">
          <span>${enabledGateways.razorpay.icon}</span> Pay with ${enabledGateways.razorpay.name}
        </button>
      `;
    }

    if (enabledGateways.stripe) {
      buttonsHTML += `
        <button class="payment-btn" data-gateway="stripe"
                style="padding: 16px; background: ${gatewayColors.stripe}; color: white; border: none; border-radius: 8px; cursor: pointer; font-size: 14px; font-weight: 600; display: flex; align-items: center; justify-content: center; gap: 8px;">
          <span>${enabledGateways.stripe.icon}</span> Pay with ${enabledGateways.stripe.name}
        </button>
      `;
    }

    if (enabledGateways.paypal) {
      buttonsHTML += `
        <button class="payment-btn" data-gateway="paypal"
                style="padding: 16px; background: ${gatewayColors.paypal}; color: white; border: none; border-radius: 8px; cursor: pointer; font-size: 14px; font-weight: 600; display: flex; align-items: center; justify-content: center; gap: 8px;">
          <span>${enabledGateways.paypal.icon}</span> Pay with ${enabledGateways.paypal.name}
        </button>
      `;
    }

    modal.innerHTML = `
      <div style="background: white; padding: 24px; border-radius: 12px; max-width: 400px; width: 100%;">
        <h3 style="margin: 0 0 16px 0; color: #1f2937;">Choose Payment Method</h3>
        <p style="margin: 0 0 24px 0; color: #6b7280; font-size: 14px;">
          Selected Plan: <strong>${planName}</strong>
        </p>
        
        <div class="payment-options" style="display: flex; flex-direction: column; gap: 12px;">
          ${buttonsHTML}
          
          <button class="payment-cancel-btn"
                  style="padding: 12px; background: #f3f4f6; color: #374151; border: none; border-radius: 8px; cursor: pointer; font-size: 14px; margin-top: 8px;">
            Cancel
          </button>
        </div>
      </div>
    `;

    document.body.appendChild(modal);

    // Attach event listeners
    modal.querySelectorAll('.payment-btn').forEach(btn => {
      const gateway = btn.getAttribute('data-gateway');
      btn.addEventListener('click', () => processPayment(planId, planName, gateway));
    });

    modal.querySelector('.payment-cancel-btn')?.addEventListener('click', closePaymentModal);
  } catch (error) {
    console.error('Error loading payment gateways:', error);
    showToast('Failed to load payment methods. Please try again.', 'error');
  }
}

/**
 * Close payment modal
 */
function closePaymentModal() {
  const modal = document.getElementById('paymentGatewayModal');
  if (modal) {
    modal.remove();
  }
}

/**
 * Process payment with selected gateway
 */
async function processPayment(planId, planName, gateway) {
  console.log(`💳 Processing payment for ${planName} via ${gateway}`);
  closePaymentModal();

  try {
    showToast(`Initiating ${gateway} payment...`, 'info');

    if (gateway === 'razorpay') {
      await processRazorpayPayment(planId, planName);
    } else if (gateway === 'stripe') {
      await processStripePayment(planId, planName);
    } else if (gateway === 'paypal') {
      await processPayPalPayment(planId, planName);
    }
  } catch (error) {
    console.error('Payment error:', error);
    showToast(`Payment failed: ${error.message}`, 'error');
  }
}

/**
 * Process Razorpay payment
 */
async function processRazorpayPayment(planId, planName) {
  try {
    showToast('Opening Razorpay checkout...', 'info');
    addLog('INFO', 'Opening Razorpay payment page...');
    await openPaymentPage(planId, 'razorpay');
  } catch (error) {
    console.error('Razorpay payment error:', error);
    showToast(`Payment failed: ${error.message}`, 'error');
  }
}

/**
 * Process PayPal payment
 */
async function processPayPalPayment(planId, planName) {
  try {
    showToast('Opening PayPal checkout...', 'info');
    addLog('INFO', 'Opening PayPal payment page...');
    await openPaymentPage(planId, 'paypal');
  } catch (error) {
    console.error('PayPal payment error:', error);
    showToast(`Payment failed: ${error.message}`, 'error');
  }
}

/**
 * Process Stripe payment
 */
async function processStripePayment(planId, planName) {
  try {
    showToast('Opening Stripe checkout...', 'info');
    addLog('INFO', 'Opening Stripe payment page...');
    await openPaymentPage(planId, 'stripe');
  } catch (error) {
    console.error('Stripe payment error:', error);
    showToast(`Payment failed: ${error.message}`, 'error');
  }
}

/**
 * Process Payoneer payment
 */
async function processPayoneerPayment(planId, planName) {
  try {
    showToast('Opening Payoneer payment...', 'info');
    addLog('INFO', 'Opening Payoneer payment page...');
    await openPaymentPage(planId, 'payoneer');
  } catch (error) {
    console.error('Payoneer payment error:', error);
    showToast(`Payment failed: ${error.message}`, 'error');
  }
}

/**
 * Open payment page in new tab with secure payment token
 */
async function openPaymentPage(planId, gateway) {
  // Check if user is logged in and has token
  if (!api.token) {
    await api.loadToken();
  }

  if (!api.token) {
    throw new Error('Please login first');
  }

  try {
    // 🔐 Call backend to create secure payment session
    console.log('🔐 Creating secure payment session...');
    showToast('Creating secure payment session...', 'info');

    const sessionResponse = await api.request('/payments/create-payment-session', {
      method: 'POST',
      body: JSON.stringify({
        planId: planId,
        gateway: gateway
      })
    });

    if (!sessionResponse.success) {
      throw new Error(sessionResponse.message || 'Failed to create payment session');
    }

    console.log('✅ Secure payment URL created:', sessionResponse.paymentUrl);

    // Open payment page with secure token (15 minutes validity)
    chrome.tabs.create({ url: sessionResponse.paymentUrl }, (tab) => {
      console.log(`✅ Payment page opened for ${gateway}:`, tab.id);
      addLog('SUCCESS', `Secure payment page opened. Token valid for 15 minutes.`);
      showToast('Complete payment in the new tab (15 min validity)', 'success', 5000);
    });

  } catch (error) {
    console.error('❌ Error creating payment session:', error);
    showToast(`Failed to create payment session: ${error.message}`, 'error', 5000);
    throw error;
  }
}

/**
 * Simulate payment success (for testing - calls backend to actually update plan)
 */
async function simulatePaymentSuccess(planId, planName, gateway) {
  try {
    showToast('Payment successful! Updating your plan...', 'success');

    console.log('💳 Processing payment for plan:', planId, planName, gateway);
    console.log('🔑 Auth token exists:', api.token ? 'Yes' : 'No');
    console.log('👤 User logged in:', state.isLoggedIn);

    // Ensure token is loaded
    if (!api.token) {
      await api.loadToken();
      console.log('🔄 Token reloaded:', api.token ? 'Yes' : 'No');
    }

    if (!api.token) {
      throw new Error('Not logged in. Please login first.');
    }

    // Call backend to actually update the plan
    let updateResult;

    if (gateway === 'razorpay') {
      // Call Razorpay verify endpoint
      console.log('📞 Calling Razorpay verify API...');
      updateResult = await api.request('/payments/razorpay/verify', {
        method: 'POST',
        body: JSON.stringify({
          razorpay_order_id: `order_${Date.now()}`,
          razorpay_payment_id: `pay_${Date.now()}`,
          razorpay_signature: 'simulated_signature',
          planId: planId
        })
      });
    } else if (gateway === 'paypal') {
      // Call PayPal capture endpoint
      console.log('📞 Calling PayPal capture API...');
      updateResult = await api.request('/payments/paypal/capture', {
        method: 'POST',
        body: JSON.stringify({
          orderId: `PAYPAL_${Date.now()}`,
          planId: planId
        })
      });
    } else if (gateway === 'payoneer') {
      // For Payoneer, we'll use PayPal capture as fallback (since backend doesn't have verify endpoint for Payoneer)
      console.log('📞 Calling Payoneer (PayPal) capture API...');
      updateResult = await api.request('/payments/paypal/capture', {
        method: 'POST',
        body: JSON.stringify({
          orderId: `PAYONEER_${Date.now()}`,
          planId: planId
        })
      });
    }

    console.log('✅ Payment API response:', updateResult);

    if (updateResult && updateResult.success) {
      // Reload user data from backend to get updated plan
      await syncUserStateFromBackend();

      showToast(`✅ Successfully upgraded to ${planName}!`, 'success', 5000);
      addLog('SUCCESS', `Upgraded to ${planName} via ${gateway}`);
    } else {
      const errorMsg = updateResult?.message || 'Failed to update plan';
      console.error('❌ Payment update failed:', errorMsg);
      throw new Error(errorMsg);
    }
  } catch (error) {
    console.error('💥 Payment error details:', error);
    const errorMessage = error.message || 'Unknown error occurred';
    showToast(`Payment Error: ${errorMessage}`, 'error', 6000);
    addLog('ERROR', `Payment failed: ${errorMessage}`);
  }
}

// Sync user state from backend (fetch fresh data)
async function syncUserStateFromBackend() {
  try {
    const result = await api.getCurrentUser();
    if (result.success && result.user) {
      state.userPlan = result.user.plan || DEFAULT_PLAN_NAME;
      state.planLimits = normalizePlanLimits(state.userPlan, result.user.planLimits);
      state.repliesThisMonth = result.user.repliesThisMonth || 0;
      state.fiverrUsername = result.user.fiverrUsername || '';
      state.timeInterval = result.user.timeInterval || 30;
      state.botStatus = result.user.botStatus || 'Offline';

      // ✨ NEW: Save subscription info
      if (result.user.subscription) {
        state.subscription = {
          status: result.user.subscription.status || 'trial',
          startDate: result.user.subscription.startDate,
          endDate: result.user.subscription.endDate
        };
      }

      await saveState();
      updateUI();
    }
  } catch (error) {
    console.error('Error syncing user state:', error);
    throw error;
  }
}

// ========================================
// END PLANS & BILLING FUNCTIONS
// ========================================

// Initialize extension
document.addEventListener('DOMContentLoaded', async () => {
  // Ensure API is initialized before anything else
  await api.init();
  console.log('✅ API initialized, token:', api.token ? 'Loaded' : 'Not found');

  await loadState();
  initializeEventListeners();
  await checkBackendConnection();
  updateUI();

  // Load announcement after everything else
  await loadAnnouncement();
});

// Check if backend is running
async function checkBackendConnection() {
  try {
    const response = await fetch('https://fivreply-production.up.railway.app/api/health');
    if (response.ok) {
      addLog('INFO', 'Connected to backend server');
    }
  } catch (error) {
    addLog('WARNING', 'Backend server not running. Please start the backend.');
  }
}

// Load state from chrome storage
async function loadState() {
  return new Promise((resolve) => {
    chrome.storage.local.get(['appState', 'authToken', 'backendUrl'], async (result) => {
      if (result.appState) {
        // Load state from storage but DON'T overwrite planLimits yet
        // We'll get fresh planLimits from backend
        const { planLimits: _, ...stateWithoutPlanLimits } = result.appState;
        state = { ...state, ...stateWithoutPlanLimits };
        console.log('📂 State loaded from Chrome storage (planLimits will be fetched from backend)');
      }

      // Load backend URL
      if (result.backendUrl) {
        state.backendUrl = result.backendUrl;
      }

      // If token exists, fetch user data from backend
      if (result.authToken && api) {
        try {
          const userData = await api.getUser();
          if (userData.success && userData.user) {
            state.isLoggedIn = true;
            state.userEmail = userData.user.email;
            state.fiverrUsername = userData.user.fiverrUsername || '';
            state.timeInterval = userData.user.timeInterval || 30;
            state.userPlan = userData.user.plan || DEFAULT_PLAN_NAME;
            state.botStatus = userData.user.botStatus || 'Offline';
            state.planLimits = normalizePlanLimits(state.userPlan, userData.user.planLimits);
            state.repliesThisMonth = userData.user.repliesThisMonth || 0;

            // ✨ NEW: Load subscription info
            if (userData.user.subscription) {
              state.subscription = {
                status: userData.user.subscription.status || 'trial',
                startDate: userData.user.subscription.startDate,
                endDate: userData.user.subscription.endDate
              };
            }

            console.log('✅ Fresh plan limits from backend:', state.planLimits);
            // ⚠️ DON'T load analytics from User model - it's in separate Analytics collection

            // Fetch user's gigs from database
            try {
              const gigsData = await api.getGigs();
              if (gigsData.success && gigsData.gigs) {
                state.gigs = gigsData.gigs;
                console.log(`✅ Loaded ${gigsData.gigs.length} gig(s) from MongoDB`);

                // Log deliverables for each gig
                gigsData.gigs.forEach((gig, index) => {
                  console.log(`  Gig #${index + 1}: ${gig.title}`);
                  console.log(`    - Deliverables:`, gig.deliverables);
                  console.log(`    - Requirements:`, gig.requirements_list);
                });
              }
            } catch (gigError) {
              console.error('Error loading gigs:', gigError);
            }

            // Fetch analytics from Analytics API (separate collection)
            try {
              const analyticsData = await api.getAnalytics();
              if (analyticsData.success && analyticsData.analytics) {
                state.analytics = {
                  totalReplies: analyticsData.analytics.totalReplies || 0,
                  repliesToday: analyticsData.analytics.today.replies || 0,
                  avgTime: Math.round(analyticsData.analytics.overall?.avgResponseTime || analyticsData.analytics.avgResponseTime || 0),
                  successRate: Math.round(analyticsData.analytics.overall?.successRate || analyticsData.analytics.overallSuccessRate || 100),
                  activeChats: analyticsData.analytics.totalUniqueCustomers || 0,  // ✅ Lifetime unique customers (Analytics tab)
                  activeConversations: analyticsData.analytics.today.uniqueCustomersCount || 0  // ✅ Today's unique customers (Bot Controls)
                };
                console.log('📊 Analytics loaded on startup:', state.analytics);
              }
            } catch (analyticsError) {
              console.error('Error loading analytics on startup:', analyticsError);
              // Keep default analytics if error
            }
          }
        } catch (error) {
          console.error('Error loading user data:', error);
        }
      }

      resolve();
    });
  });
}

// Save state to chrome storage
function saveState() {
  // ✅ Save entire state including analytics
  chrome.storage.local.set({
    appState: state,
    backendUrl: state.backendUrl
  });

  console.log('💾 State saved to Chrome storage (analytics preserved):', {
    totalReplies: state.analytics.totalReplies,
    repliesToday: state.analytics.repliesToday,
    activeChats: state.analytics.activeChats
  });
}

// Clear all user-specific data
async function clearUserData(keepToken = false) {
  console.log('🗑️ Clearing all user-specific data...');

  // Save current token if needed (for login/signup flow)
  const currentToken = keepToken ? api.token : null;

  // Reset state to default
  state = {
    isLoggedIn: false,
    userEmail: '',
    userPlan: DEFAULT_PLAN_NAME,
    fiverrUsername: '',
    timeInterval: null, // ✨ User must set this
    botStatus: 'Offline',
    backendUrl: state.backendUrl || 'https://fivreply-production.up.railway.app', // Keep backend URL
    gigs: [],
    analytics: {
      totalReplies: 0,
      repliesToday: 0,  // ✅ Reset daily count
      avgTime: 0,
      successRate: 0,
      activeChats: 0,
      activeConversations: 0
    },
    logs: [],
    planLimits: getPlanLimitsFor(DEFAULT_PLAN_NAME),
  };

  // Clear Chrome storage (keep backend URL and optionally token)
  const backendUrl = state.backendUrl;
  await chrome.storage.local.clear();
  await chrome.storage.local.set({ backendUrl });

  // Restore token if kept
  if (keepToken && currentToken) {
    api.setToken(currentToken);
  } else {
    // Clear API token
    if (api) {
      api.clearToken();
    }
  }

  console.log('✅ All user data cleared');
}

// Initialize all event listeners
function initializeEventListeners() {
  // Announcement Bar
  document.getElementById('closeAnnouncement')?.addEventListener('click', () => {
    dismissAnnouncement();
  });

  // Refresh Mode Toggle
  document.getElementById('refreshModeToggle')?.addEventListener('change', handleRefreshModeToggle);

  // Login/Signup Modal Controls
  document.getElementById('showLoginBtn')?.addEventListener('click', showLoginModal);
  document.getElementById('loginBtn')?.addEventListener('click', handleLogin);
  document.getElementById('modalClose')?.addEventListener('click', hideLoginModal);
  document.getElementById('modalOverlay')?.addEventListener('click', hideLoginModal);

  // Signup Modal Controls
  document.getElementById('showSignupLink')?.addEventListener('click', showSignupModal);
  document.getElementById('signupBtn')?.addEventListener('click', handleSignup);
  document.getElementById('signupClose')?.addEventListener('click', hideSignupModal);
  document.getElementById('signupOverlay')?.addEventListener('click', hideSignupModal);
  document.getElementById('showLoginLink')?.addEventListener('click', () => {
    hideSignupModal();
    showLoginModal();
  });

  // Logout
  document.getElementById('logoutBtn')?.addEventListener('click', handleLogout);

  // Dashboard controls
  document.getElementById('saveSettingsBtn')?.addEventListener('click', handleSaveSettings);
  document.getElementById('startBotBtn')?.addEventListener('click', handleStartBot);
  document.getElementById('stopBotBtn')?.addEventListener('click', handleStopBot);

  // Gigs
  document.getElementById('saveGigsBtn')?.addEventListener('click', handleSaveGigs);

  // Plans & Upgrade
  document.getElementById('upgradeBtn')?.addEventListener('click', () => switchTab('payment'));

  // Tabs
  document.querySelectorAll('.tab').forEach(tab => {
    tab.addEventListener('click', (e) => {
      const tabName = e.currentTarget.getAttribute('data-tab');
      switchTab(tabName);
    });
  });

  // WhatsApp Contact Button
  document.getElementById('whatsappContactBtn')?.addEventListener('click', handleWhatsAppContact);

  // ✨ Password Change & Forgot Password Modals
  document.getElementById('showForgotPasswordLink')?.addEventListener('click', showForgotPasswordModal);
  document.getElementById('forgotPasswordClose')?.addEventListener('click', hideForgotPasswordModal);
  document.getElementById('forgotPasswordOverlay')?.addEventListener('click', hideForgotPasswordModal);
  document.getElementById('sendResetLinkBtn')?.addEventListener('click', handleForgotPassword);
  document.getElementById('backToLoginLink')?.addEventListener('click', () => {
    hideForgotPasswordModal();
    showLoginModal();
  });

  document.getElementById('showChangePasswordBtn')?.addEventListener('click', showChangePasswordModal);
  document.getElementById('changePasswordClose')?.addEventListener('click', hideChangePasswordModal);
  document.getElementById('changePasswordOverlay')?.addEventListener('click', hideChangePasswordModal);
  document.getElementById('changePasswordBtn')?.addEventListener('click', handleChangePassword);

}

// Handle WhatsApp Contact
function handleWhatsAppContact() {
  // Replace with your WhatsApp number (in international format without + or spaces)
  // Example: '8801234567890' for Bangladesh number
  const phoneNumber = '919800248298'; // আপনার WhatsApp number এখানে দিন
  const message = encodeURIComponent('Hi! I need help with Fiverr Auto Reply Extension.');

  // Open WhatsApp in new tab
  const whatsappUrl = `https://wa.me/${phoneNumber}?text=${message}`;
  chrome.tabs.create({ url: whatsappUrl });

  addLog('INFO', 'Opening WhatsApp contact...');
  showToast('Opening WhatsApp', 'Redirecting to WhatsApp chat...', 'info');
}

function showLoginModal() {
  const modal = document.getElementById('loginModal');
  modal?.classList.remove('hidden');
}

// Hide login modal
function hideLoginModal() {
  const modal = document.getElementById('loginModal');
  modal?.classList.add('hidden');
}

// Show signup modal
function showSignupModal() {
  hideLoginModal();
  const modal = document.getElementById('signupModal');
  modal?.classList.remove('hidden');
}

// Hide signup modal
function hideSignupModal() {
  const modal = document.getElementById('signupModal');
  modal?.classList.add('hidden');
}

// ==================== PASSWORD MODALS ====================

// Show Forgot Password Modal
function showForgotPasswordModal() {
  const modal = document.getElementById('forgotPasswordModal');
  if (modal) {
    modal.classList.remove('hidden');
    // Clear input
    const emailInput = document.getElementById('forgotPasswordEmail');
    if (emailInput) emailInput.value = '';
  }
  hideLoginModal(); // Hide login modal
}

// Hide Forgot Password Modal
function hideForgotPasswordModal() {
  const modal = document.getElementById('forgotPasswordModal');
  if (modal) {
    modal.classList.add('hidden');
  }
}

// Show Change Password Modal
function showChangePasswordModal() {
  if (!state.isLoggedIn) {
    showToast('Login Required', 'Please login first to change password', 'warning');
    return;
  }

  const modal = document.getElementById('changePasswordModal');
  if (modal) {
    modal.classList.remove('hidden');
    // Clear inputs
    const currentPassword = document.getElementById('currentPassword');
    const newPassword = document.getElementById('newPassword');
    const confirmNewPassword = document.getElementById('confirmNewPassword');
    if (currentPassword) currentPassword.value = '';
    if (newPassword) newPassword.value = '';
    if (confirmNewPassword) confirmNewPassword.value = '';
  }
}

// Hide Change Password Modal
function hideChangePasswordModal() {
  const modal = document.getElementById('changePasswordModal');
  if (modal) {
    modal.classList.add('hidden');
  }
}

// Handle Forgot Password
async function handleForgotPassword() {
  const emailInput = document.getElementById('forgotPasswordEmail');
  const email = emailInput?.value?.trim();

  if (!email) {
    showToast('Email Required', 'Please enter your email address', 'warning');
    return;
  }

  const btn = document.getElementById('sendResetLinkBtn');
  if (!btn) return;

  btn.disabled = true;
  btn.textContent = 'Sending...';

  try {
    const response = await fetch(`${state.backendUrl}/api/auth/forgot-password`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({ email })
    });

    const data = await response.json();

    if (data.success) {
      showToast('Email Sent', data.message, 'success', 6000);
      hideForgotPasswordModal();
      addLog('SUCCESS', '✅ Password reset instructions sent');
    } else {
      throw new Error(data.message || 'Failed to send reset link');
    }
  } catch (error) {
    console.error('Forgot password error:', error);
    showToast('Error', error.message || 'Failed to send reset link', 'error');
    addLog('ERROR', `❌ Forgot password failed: ${error.message}`);
  } finally {
    btn.disabled = false;
    btn.innerHTML = `
      <svg class="btn-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
        <path d="M4 4h16c1.1 0 2 .9 2 2v12c0 1.1-.9 2-2 2H4c-1.1 0-2-.9-2-2V6c0-1.1.9-2 2-2z"></path>
        <polyline points="22,6 12,13 2,6"></polyline>
      </svg>
      Send Reset Link
    `;
  }
}

// Handle Change Password
async function handleChangePassword() {
  const currentPassword = document.getElementById('currentPassword')?.value;
  const newPassword = document.getElementById('newPassword')?.value;
  const confirmPassword = document.getElementById('confirmNewPassword')?.value;

  // Validation
  if (!currentPassword || !newPassword || !confirmPassword) {
    showToast('All Fields Required', 'Please fill in all password fields', 'warning');
    return;
  }

  if (newPassword.length < 6) {
    showToast('Invalid Password', 'New password must be at least 6 characters', 'warning');
    return;
  }

  if (newPassword !== confirmPassword) {
    showToast('Passwords Do Not Match', 'New password and confirm password must match', 'error');
    return;
  }

  const btn = document.getElementById('changePasswordBtn');
  if (!btn) return;

  btn.disabled = true;
  btn.textContent = 'Changing...';

  try {
    // Call API to change password
    const result = await api.request('/auth/change-password', {
      method: 'POST',
      body: JSON.stringify({
        currentPassword,
        newPassword
      })
    });

    if (result.success) {
      showToast('Success!', 'Password changed successfully', 'success');
      hideChangePasswordModal();
      addLog('SUCCESS', '✅ Password changed successfully');
    } else {
      throw new Error(result.message || 'Failed to change password');
    }
  } catch (error) {
    console.error('Change password error:', error);
    showToast('Error', error.message || 'Failed to change password', 'error');
    addLog('ERROR', `❌ Password change failed: ${error.message}`);
  } finally {
    btn.disabled = false;
    btn.innerHTML = `
      <svg class="btn-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
        <path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"></path>
        <polyline points="22 4 12 14.01 9 11.01"></polyline>
      </svg>
      Change Password
    `;
  }
}

// ==================== END PASSWORD MODALS ====================


// Handle Sign Up
async function handleSignup() {
  const email = document.getElementById('signupEmail').value;
  const password = document.getElementById('signupPassword').value;
  const confirmPassword = document.getElementById('signupConfirmPassword').value;
  const fiverrUsername = document.getElementById('signupFiverrUsername').value.trim();

  // Validation
  if (!email || !password || !confirmPassword || !fiverrUsername) {
    showToast('Validation Error', 'Please fill in all fields', 'error');
    addLog('ERROR', 'Please fill in all fields');
    return;
  }

  if (password !== confirmPassword) {
    showToast('Password Mismatch', 'Passwords do not match', 'error');
    addLog('ERROR', 'Passwords do not match');
    return;
  }

  if (password.length < 6) {
    showToast('Invalid Password', 'Password must be at least 6 characters', 'error');
    addLog('ERROR', 'Password must be at least 6 characters');
    return;
  }

  // Disable button during request
  const signupBtn = document.getElementById('signupBtn');
  const originalText = signupBtn.textContent;
  signupBtn.disabled = true;
  signupBtn.textContent = 'Creating Account...';

  try {
    showToast('Creating Account', 'Please wait...', 'info', 2000);
    addLog('INFO', 'Creating account...');

    // Clear old user data BEFORE signup
    await clearUserData();

    const result = await api.register(email, password, {
      fiverrUsername
    });

    if (result.success) {
      // Success! Save new user data and update UI
      state.isLoggedIn = true;
      state.userEmail = result.user.email;
      state.userPlan = result.user.plan || 'Free';
      state.fiverrUsername = result.user.fiverrUsername || fiverrUsername;
      state.gigs = []; // Start with empty gigs
      state.planLimits = normalizePlanLimits(state.userPlan, result.user.planLimits);
      state.repliesThisMonth = result.user.repliesThisMonth || 0;

      // ✨ NEW: Load subscription info
      if (result.user.subscription) {
        state.subscription = {
          status: result.user.subscription.status || 'trial',
          startDate: result.user.subscription.startDate,
          endDate: result.user.subscription.endDate
        };
      }

      // Token is already set by api.register()
      console.log('✅ Token set:', api.token ? 'Yes' : 'No');
      console.log('✅ Plan limits from backend:', result.user.planLimits);
      console.log('✅ Normalized plan limits:', state.planLimits);

      saveState();

      // Show success message
      showToast(
        'Account Created Successfully! 🎉',
        `Welcome ${email}! Your account has been created and saved to database.`,
        'success',
        5000
      );
      addLog('SUCCESS', 'Account created successfully! Data saved to MongoDB.');

      // Hide modal and show dashboard
      hideSignupModal();
      updateUI();

      // Clear form
      document.getElementById('signupEmail').value = '';
      document.getElementById('signupPassword').value = '';
      document.getElementById('signupConfirmPassword').value = '';
      document.getElementById('signupFiverrUsername').value = '';
    } else {
      // Backend returned success: false
      showToast(
        'Registration Failed',
        result.message || 'Failed to create account',
        'error'
      );
      addLog('ERROR', result.message || 'Failed to create account');
    }
  } catch (error) {
    console.error('Signup error:', error);

    // Network or other error
    let errorMessage = 'Failed to create account. ';

    if (error.message.includes('Failed to fetch') || error.message.includes('NetworkError')) {
      errorMessage += 'Backend server is not running. Please start the backend (npm start in backend folder).';
    } else if (error.message.includes('already exists')) {
      errorMessage += 'This email is already registered. Please sign in instead.';
    } else {
      errorMessage += error.message || 'Unknown error occurred.';
    }

    showToast('Registration Error', errorMessage, 'error', 6000);
    addLog('ERROR', errorMessage);
  } finally {
    // Re-enable button
    signupBtn.disabled = false;
    signupBtn.textContent = originalText;
  }
}

// Handle Login
async function handleLogin() {
  const email = document.getElementById('loginEmail').value;
  const password = document.getElementById('loginPassword').value;

  // Validation
  if (!email || !password) {
    showToast('Validation Error', 'Please enter email and password', 'error');
    addLog('ERROR', 'Please enter email and password');
    return;
  }

  // Disable button during request
  const loginBtn = document.getElementById('loginBtn');
  const originalText = loginBtn.textContent;
  loginBtn.disabled = true;
  loginBtn.textContent = 'Signing In...';

  try {
    showToast('Signing In', 'Please wait...', 'info', 2000);
    addLog('INFO', 'Logging in...');

    // Clear old user data BEFORE login
    await clearUserData();

    const result = await api.login(email, password);

    if (result.success) {
      // Success! Load NEW user data
      state.isLoggedIn = true;
      state.userEmail = result.user.email;
      state.fiverrUsername = result.user.fiverrUsername || '';
      state.timeInterval = result.user.timeInterval || 30;
      state.userPlan = result.user.plan || DEFAULT_PLAN_NAME;
      state.botStatus = result.user.botStatus || 'Offline';
      state.planLimits = normalizePlanLimits(state.userPlan, result.user.planLimits);
      state.repliesThisMonth = result.user.repliesThisMonth || 0;

      // ✨ NEW: Load subscription info
      if (result.user.subscription) {
        state.subscription = {
          status: result.user.subscription.status || 'trial',
          startDate: result.user.subscription.startDate,
          endDate: result.user.subscription.endDate
        };
      }

      // Token is already set by api.login()
      console.log('✅ Token set:', api.token ? 'Yes' : 'No');

      // Load analytics from database
      try {
        const analyticsData = await api.getAnalytics();
        if (analyticsData.success && analyticsData.analytics) {
          state.analytics = {
            totalReplies: analyticsData.analytics.totalReplies || 0,
            repliesToday: analyticsData.analytics.today.replies || 0,
            avgTime: Math.round(analyticsData.analytics.overall?.avgResponseTime || analyticsData.analytics.avgResponseTime || 0),
            successRate: Math.round(analyticsData.analytics.overall?.successRate || analyticsData.analytics.overallSuccessRate || 100),
            activeChats: analyticsData.analytics.totalUniqueCustomers || 0,  // ✅ Lifetime unique customers (Analytics tab)
            activeConversations: analyticsData.analytics.today.uniqueCustomersCount || 0  // ✅ Today's unique customers (Bot Controls)
          };
          addLog('SUCCESS', `✅ Analytics loaded: ${state.analytics.repliesToday} replies today, ${state.analytics.totalReplies} total`);
          console.log('📊 Analytics from database:', state.analytics);
        } else {
          // Fallback to default
          state.analytics = {
            totalReplies: 0,
            repliesToday: 0,
            avgTime: 0,
            successRate: 100,
            activeChats: 0,
            activeConversations: 0
          };
          addLog('INFO', 'No analytics found, starting fresh');
        }
      } catch (analyticsError) {
        console.error('Error loading analytics:', analyticsError);
        state.analytics = {
          totalReplies: 0,
          repliesToday: 0,
          avgTime: 0,
          successRate: 100,
          activeChats: 0,
          activeConversations: 0
        };
        addLog('WARNING', 'Could not load analytics from database');
      }

      // Fetch THIS user's gigs from database
      try {
        const gigsData = await api.getGigs();
        if (gigsData.success && gigsData.gigs) {
          state.gigs = gigsData.gigs;
          addLog('SUCCESS', `✅ Loaded ${gigsData.gigs.length} gig(s) for ${result.user.email}`);

          // Log deliverables for each gig
          console.log(`📦 Loaded ${gigsData.gigs.length} gig(s) for user: ${result.user.email}`);
          gigsData.gigs.forEach((gig, index) => {
            console.log(`  Gig #${index + 1}: ${gig.title}`);
            console.log(`    - Link: ${gig.link}`);
            console.log(`    - Deliverables:`, gig.deliverables);
            console.log(`    - Requirements:`, gig.requirements_list);
          });
        } else {
          // No gigs for this user
          state.gigs = [];
          addLog('INFO', 'No gigs found for this user');
        }
      } catch (gigError) {
        console.error('Error loading gigs:', gigError);
        addLog('WARNING', 'Could not load gigs from database');
        state.gigs = [];
      }

      saveState();

      // Show success message
      showToast(
        'Welcome Back! 👋',
        `Successfully signed in as ${email}. Your data loaded from database.`,
        'success',
        5000
      );
      addLog('SUCCESS', 'Logged in successfully! Data loaded from MongoDB.');

      // Hide modal and show dashboard
      hideLoginModal();
      updateUI();

      // Clear form
      document.getElementById('loginEmail').value = '';
      document.getElementById('loginPassword').value = '';
    } else {
      // Backend returned success: false
      showToast(
        'Sign In Failed',
        result.message || 'Invalid credentials',
        'error'
      );
      addLog('ERROR', result.message || 'Invalid credentials');
    }
  } catch (error) {
    console.error('Login error:', error);

    // Network or other error
    let errorMessage = 'Failed to sign in. ';

    if (error.message.includes('Failed to fetch') || error.message.includes('NetworkError')) {
      errorMessage += 'Backend server is not running. Please start the backend (npm start in backend folder).';
    } else if (error.message.includes('Invalid credentials') || error.message.includes('401')) {
      errorMessage += 'Invalid email or password. Please check your credentials.';
    } else {
      errorMessage += error.message || 'Unknown error occurred.';
    }

    showToast('Sign In Error', errorMessage, 'error', 6000);
    addLog('ERROR', errorMessage);
  } finally {
    // Re-enable button
    loginBtn.disabled = false;
    loginBtn.textContent = originalText;
  }
}

// Handle logout
async function handleLogout() {
  try {
    // Stop bot first if running
    if (state.botStatus === 'Online') {
      console.log('🛑 Stopping bot before logout...');

      // Update bot status to offline
      if (state.isLoggedIn) {
        await api.updateBotStatus('Offline').catch(() => {
          // Ignore error if API fails
        });
      }

      // Send stop message to background
      chrome.runtime.sendMessage({
        action: 'stopBot'
      }).catch(err => {
        console.warn('Background script not available:', err);
      });

      // Force stop all activities on Fiverr page
      chrome.tabs.query({ url: '*://*.fiverr.com/*' }, (tabs) => {
        tabs.forEach(tab => {
          chrome.tabs.sendMessage(tab.id, {
            action: 'forceStopBot'
          }).catch(() => {
            // Ignore if content script not loaded
          });
        });
      });
    }

    // Clear all user data and reload extension to ensure clean state
    await clearUserData();
    addLog('INFO', 'Logged out successfully - Bot stopped, all data cleared');
    showToast('Logged Out', 'Bot stopped and you have been logged out successfully', 'success');
    chrome.runtime.reload();
  } catch (error) {
    console.error('Logout error:', error);
    addLog('ERROR', 'Logout failed: ' + error.message);
  }
}

// Handle save settings (username + interval)
async function handleSaveSettings() {
  const username = document.getElementById('fiverrUsername')?.value?.trim();
  const interval = parseInt(document.getElementById('timeInterval')?.value);

  // ✅ Validate username using security helper
  const usernameValidation = window.SecurityHelpers.validateUsername(username);
  if (!usernameValidation.valid) {
    showToast('Invalid Username', usernameValidation.error, 'error');
    addLog('ERROR', usernameValidation.error);
    return;
  }

  // ✅ Validate interval using security helper
  const intervalValidation = window.SecurityHelpers.validateTimeInterval(interval);
  if (!intervalValidation.valid) {
    showToast('Invalid Interval', intervalValidation.error, 'error');
    addLog('ERROR', intervalValidation.error);
    return;
  }

  // Use sanitized values
  const sanitizedUsername = usernameValidation.sanitized;
  const sanitizedInterval = intervalValidation.sanitized;

  if (!state.isLoggedIn) {
    showToast('Not Logged In', 'Please sign in first to save settings', 'warning');
    addLog('ERROR', 'Please login first');
    showLoginModal();
    return;
  }

  // Disable button during request
  const saveBtn = document.getElementById('saveSettingsBtn');
  const originalText = saveBtn ? saveBtn.textContent : 'Save Settings';
  if (saveBtn) {
    saveBtn.disabled = true;
    saveBtn.textContent = 'Saving...';
  }

  try {
    showToast('Saving Settings', 'Please wait...', 'info', 2000);
    addLog('INFO', 'Saving username and settings...');

    const result = await api.updateProfile(sanitizedUsername, sanitizedInterval);

    if (result.success) {
      // Success! Save data locally
      const oldInterval = state.timeInterval;
      state.fiverrUsername = sanitizedUsername;
      state.timeInterval = sanitizedInterval;
      saveState();

      // ✨ CRITICAL FIX: Always update interval in background script
      // This ensures botState.interval is updated even if refresh mode is off
      chrome.runtime.sendMessage({
        action: 'setInterval',
        interval: sanitizedInterval
      }).catch(err => {
        console.warn('Background script not available:', err);
      });

      // ✨ If refresh mode is ON and interval changed, also update refresh mode
      if (state.refreshMode && oldInterval !== sanitizedInterval) {
        console.log(`🔄 Interval changed (${oldInterval}s → ${sanitizedInterval}s) - restarting refresh mode`);
        addLog('INFO', `Restarting refresh mode with new interval (${sanitizedInterval}s)`);

        // Send updated interval to refresh mode
        chrome.runtime.sendMessage({
          action: 'setRefreshMode',
          enabled: true,
          interval: sanitizedInterval
        }).catch(err => {
          console.warn('Background script not available:', err);
        });

        showToast(
          'Settings Saved & Refresh Mode Updated! ✅',
          `New interval: ${interval}s. Refresh mode restarted.`,
          'success',
          5000
        );
      } else {
        // Show username change info if available
        const changeInfo = result.usernameChangeInfo;
        let changeWarning = '';
        if (changeInfo && !changeInfo.allowed) {
          changeWarning = `\n\n⚠️ Note: ${changeInfo.reason}`;
        } else if (changeInfo && changeInfo.changesRemaining !== undefined) {
          changeWarning = `\n\nℹ️ You have ${changeInfo.changesRemaining} username change(s) remaining in the next 30 days.`;
        }

        // Show success message with device info
        if (result.isActiveDevice) {
          showToast(
            'Settings Saved Successfully! ✅',
            `Fiverr username "${username}" and time interval (${interval}s) saved to database. This device is now authorized to run the bot.${changeWarning}`,
            'success',
            changeWarning ? 8000 : 6000
          );
          addLog('SUCCESS', `Username saved! This device is now active. Interval: ${interval}s`);
        } else {
          showToast(
            'Settings Saved (Device Inactive) ⚠️',
            `Settings saved to database, but another device was recently activated. Save again on this device to make it active.${changeWarning}`,
            'warning',
            changeWarning ? 8000 : 6000
          );
          addLog('WARNING', 'Username saved but another device is active');
        }
      }

      updateUI();

      // Send message to background script
      chrome.runtime.sendMessage({
        action: 'registerUsername',
        username: username
      }).catch(err => {
        console.warn('Background script not available:', err);
      });
    } else {
      // Backend returned success: false
      showToast(
        'Save Error',
        result.message || 'Failed to save settings',
        'error',
        6000
      );
      addLog('ERROR', result.message || 'Failed to save settings');
    }
  } catch (error) {
    console.error('Error saving username:', error);

    // Network or other error
    let errorMessage = 'Failed to save settings. ';

    if (error.message.includes('Failed to fetch') || error.message.includes('NetworkError')) {
      errorMessage += 'Backend server is not running. Please start the backend (npm start in backend folder).';
    } else if (error.message.includes('401') || error.message.includes('Unauthorized')) {
      errorMessage += 'Session expired. Please sign in again.';
    } else {
      errorMessage += error.message || 'Unknown error occurred.';
    }

    showToast('Save Error', errorMessage, 'error', 6000);
    addLog('ERROR', errorMessage);
  } finally {
    // Re-enable button
    if (saveBtn) {
      saveBtn.disabled = false;
      saveBtn.textContent = originalText;
    }
  }
}

// Handle start bot
async function handleStartBot() {
  if (!state.isLoggedIn) {
    addLog('ERROR', 'Please login first');
    showLoginModal();
    return;
  }

  // Get username and interval from input fields
  const usernameInput = document.getElementById('fiverrUsername').value.trim();
  const intervalInput = parseInt(document.getElementById('timeInterval').value) || state.timeInterval || 30;

  // Check if username has been saved
  if (!state.fiverrUsername) {
    showToast(
      'Settings Not Saved ⚠️',
      'Please save your Fiverr username and interval time first by clicking "Save Settings" button',
      'warning',
      6000
    );
    addLog('WARNING', '⚠️ Settings not saved. Please click "Save Settings" button first');

    // Keep bot offline
    state.botStatus = 'Offline';
    saveState();
    updateUI();
    return;
  }

  // Check if input fields have unsaved changes
  if (usernameInput && usernameInput !== state.fiverrUsername) {
    showToast(
      'Unsaved Changes ⚠️',
      `Username changed to "${usernameInput}" but not saved. Please click "Save Settings" button first.`,
      'warning',
      6000
    );
    addLog('WARNING', '⚠️ Username changed but not saved. Please save first.');

    // Keep bot offline
    state.botStatus = 'Offline';
    saveState();
    updateUI();
    return;
  }

  if (intervalInput && intervalInput !== state.timeInterval) {
    showToast(
      'Unsaved Changes ⚠️',
      `Interval changed to ${intervalInput}s but not saved. Please click "Save Settings" button first.`,
      'warning',
      6000
    );
    addLog('WARNING', '⚠️ Interval changed but not saved. Please save first.');

    // Keep bot offline
    state.botStatus = 'Offline';
    saveState();
    updateUI();
    return;
  }

  // Use saved settings
  const finalUsername = state.fiverrUsername;
  const finalInterval = state.timeInterval || 30;

  // Validate interval (should already be validated during save, but double check)
  if (finalInterval < 10) {
    showToast('Invalid Interval', 'Time interval must be at least 10 seconds', 'error');
    addLog('ERROR', 'Time interval too small');

    // Keep bot offline
    state.botStatus = 'Offline';
    saveState();
    updateUI();
    return;
  }

  // Log using saved settings
  console.log(`✅ Using saved settings: ${state.fiverrUsername}, ${state.timeInterval}s`);
  addLog('INFO', `✅ Using saved settings: ${state.fiverrUsername}, ${state.timeInterval}s`);

  // Check if plan has expired before starting bot
  if (isPlanExpired()) {
    const endDate = new Date(state.subscription.endDate);
    addLog('ERROR', '⚠️ Your subscription has expired. Please upgrade your plan.');
    showToast(
      'Plan Expired ⚠️',
      `Your plan expired on ${endDate.toLocaleDateString()}. Please upgrade your plan to continue using the bot.`,
      'error',
      8000
    );

    // Keep bot offline
    state.botStatus = 'Offline';
    saveState();
    updateUI();

    // Switch to plans tab after a short delay
    setTimeout(() => {
      switchTab('payment');
      loadPlans();
    }, 2000);

    return;
  }

  // Check replies remaining before starting bot
  const effectiveLimits = getEffectivePlanLimits();
  const totalReplies = typeof effectiveLimits.repliesPerMonth === 'number' ? effectiveLimits.repliesPerMonth : -1;
  const usedReplies = state.repliesThisMonth || 0;
  const remainingReplies = totalReplies === -1 ? 'Unlimited' : Math.max(0, totalReplies - usedReplies);

  // If replies remaining is 0 or less (and not unlimited), prevent bot from starting
  if (totalReplies !== -1 && remainingReplies !== 'Unlimited' && remainingReplies <= 0) {
    addLog('ERROR', '⚠️ Replies remaining limit reached. Please upgrade your plan.');
    showToast(
      'Replies Limit Reached ⚠️',
      `You've used all ${totalReplies} replies this month. Please upgrade your plan to continue using the bot.`,
      'error',
      8000
    );

    // Keep bot offline
    state.botStatus = 'Offline';
    saveState();
    updateUI();

    // Switch to plans tab after a short delay
    setTimeout(() => {
      switchTab('payment');
      loadPlans();
    }, 2000);

    return;
  }

  try {
    // Check device authorization
    addLog('INFO', 'Checking device authorization...');
    const auth = await api.checkDeviceAuth();

    if (!auth.authorized) {
      addLog('ERROR', auth.message);
      showToast(
        'Device Not Authorized ❌',
        auth.message || 'Another device is active. Please save settings to activate this device.',
        'error',
        6000
      );

      // Keep bot offline
      state.botStatus = 'Offline';
      saveState();
      updateUI();
      return;
    }

    // Update bot status
    await api.updateBotStatus('Online');

    state.botStatus = 'Online';
    saveState();
    updateUI();
    addLog('SUCCESS', 'Bot started successfully');

    // Show success toast with saved settings
    showToast(
      'Bot Started Successfully! ✅',
      `Saved Fiverr username: "${finalUsername}" and interval time: ${finalInterval}s`,
      'success',
      5000
    );

    // Send message to background script with gigs data
    chrome.runtime.sendMessage({
      action: 'startBot',
      gigs: state.gigs || []  // Pass gigs to background
    }).catch(err => {
      console.warn('Background script not available:', err);
    });

    addLog('INFO', `📦 Sending ${state.gigs.length} gig(s) to bot`);
  } catch (error) {
    console.error('Error starting bot:', error);
    addLog('ERROR', error.message || 'Failed to start bot');

    showToast(
      'Failed to Start Bot ❌',
      error.message || 'An error occurred while starting the bot',
      'error',
      5000
    );

    // Keep bot offline
    state.botStatus = 'Offline';
    saveState();
    updateUI();
  }
}

// Handle stop bot
async function handleStopBot() {
  try {
    if (state.isLoggedIn) {
      await api.updateBotStatus('Offline');
    }

    state.botStatus = 'Offline';
    saveState();
    updateUI();
    addLog('INFO', '🛑 Bot stopped - All activities halted');

    // Send message to background script
    chrome.runtime.sendMessage({
      action: 'stopBot'
    }).catch(err => {
      console.warn('Background script not available:', err);
    });

    // Force stop all activities on Fiverr page
    chrome.tabs.query({ url: '*://*.fiverr.com/*' }, (tabs) => {
      tabs.forEach(tab => {
        chrome.tabs.sendMessage(tab.id, {
          action: 'forceStopBot'
        }).catch(() => {
          // Ignore if content script not loaded
        });
      });
    });

    showToast('Bot Stopped', 'All bot activities have been halted', 'info', 3000);
  } catch (error) {
    console.error('Error stopping bot:', error);
    addLog('ERROR', error.message || 'Failed to stop bot');
  }
}

// Handle save gigs
/**
 * Generate summaries in background for changed gigs
 * This runs asynchronously - user can close popup
 * @param {Array} savedGigs - Gigs saved in database
 * @param {Array} gigNumbersNeedingSummary - Which gig numbers need summaries
 */
async function generateGigSummariesInBackground(savedGigs, gigNumbersNeedingSummary) {
  console.log(`\n🔄 Starting background summary generation...`);
  console.log(`   Gigs to process: ${gigNumbersNeedingSummary.join(', ')}`);

  let completed = 0;
  let failed = 0;

  for (const gigNumber of gigNumbersNeedingSummary) {
    const gig = savedGigs[gigNumber - 1];

    if (!gig) {
      console.error(`❌ Gig #${gigNumber} not found in saved gigs`);
      failed++;
      continue;
    }

    try {
      console.log(`\n🤖 Generating summary for Gig #${gigNumber}: "${gig.title}"`);

      const gigInfo = {
        matched: true,
        gigNumber: gigNumber,
        title: gig.title,
        description: gig.description || '',
        requirements_list: gig.requirements_list || [],
        packages: gig.packages || {}
      };

      const summary = await generateAIGigSummary(gigInfo, gigNumber);

      if (summary) {
        console.log(`✅ Summary generated (${summary.length} chars)`);

        // Update gig in database via API
        const updateResult = await api.updateGigSummary(gig._id, summary);

        if (updateResult.success) {
          console.log(`💾 Summary saved to database for Gig #${gigNumber}`);
          completed++;

          // Update local state
          const localGig = state.gigs.find(g => g._id === gig._id);
          if (localGig) {
            localGig.summary = summary;
          }
        } else {
          console.error(`❌ Failed to save summary for Gig #${gigNumber}`);
          failed++;
        }
      } else {
        console.error(`❌ Summary generation failed for Gig #${gigNumber}`);
        failed++;
      }
    } catch (error) {
      console.error(`❌ Error processing Gig #${gigNumber}:`, error);
      failed++;
    }
  }

  console.log(`\n${'='.repeat(60)}`);
  console.log(`📊 BACKGROUND SUMMARY GENERATION COMPLETE`);
  console.log(`${'='.repeat(60)}`);
  console.log(`✅ Completed: ${completed}`);
  console.log(`❌ Failed: ${failed}`);
  console.log(`📦 Total: ${gigNumbersNeedingSummary.length}`);
  console.log(`${'='.repeat(60)}\n`);

  // Save updated state
  if (completed > 0) {
    saveState();
    console.log(`✅ State updated with ${completed} new summaries`);

    // Show notification
    showToast(`Generated ${completed} gig summaries!`, 'success');
  }
}

/**
 * Generate AI-powered gig summary via backend API
 * Input limit: 1000 chars (title, description, packages, requirements)
 * Output: 200 chars max summary
 * @param {Object} gigInfo - Gig information
 * @param {Number} gigNumber - Gig number
 * @returns {Promise<string>} AI-generated summary (200 chars max)
 */
async function generateAIGigSummary(gigInfo, gigNumber) {
  if (!gigInfo || !gigInfo.title) {
    console.log('⚠️ No gig info to generate summary');
    return '';
  }

  try {
    // ✅ Get token from chrome storage (same way api.js does)
    const token = await new Promise((resolve) => {
      chrome.storage.local.get(['authToken'], (result) => {
        resolve(result.authToken || null);
      });
    });

    if (!token) {
      console.error('❌ No auth token found. Please login again.');
      showToast('Authentication Required', 'Please logout and login again', 'error');
      return '';
    }

    console.log(`🤖 Calling AI to generate gig summary for Gig #${gigNumber}...`);
    console.log(`   Backend URL: ${state.backendUrl}`);
    console.log(`   Auth token exists: ${!!token}`);

    const response = await fetch(`${state.backendUrl}/api/ai/generate-gig-summary`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`
      },
      body: JSON.stringify({
        title: gigInfo.title,
        description: gigInfo.description || '',
        deliverables: gigInfo.deliverables || [],
        requirements_list: gigInfo.requirements_list || [],
        packages: gigInfo.packages || {},
        gigNumber: gigNumber
      })
    });

    if (!response.ok) {
      const errorData = await response.json();
      console.error('❌ AI summary generation failed:', errorData.message);
      showToast('AI Summary Failed', errorData.message || 'Could not generate summary', 'error');
      return '';
    }

    const data = await response.json();
    const summary = data.summary || '';

    console.log(`✅ AI summary generated (${summary.length} chars): "${summary}"`);
    console.log(`💰 Tokens used: ${data.tokensUsed}, Cost: $${data.usage?.estimatedCost || '0'}`);

    return summary;

  } catch (error) {
    console.error('❌ Error generating AI summary:', error);
    showToast('Error', 'Failed to generate AI summary. Check console for details.', 'error');
    return '';
  }
}

async function handleSaveGigs() {
  if (!state.isLoggedIn) {
    addLog('ERROR', 'Please login first');
    showToast('Please login first', 'error');
    showLoginModal();
    return;
  }

  const gigs = [];
  const gigElements = document.querySelectorAll('.gig-item');

  console.log('📦 Processing gigs for save...');

  gigElements.forEach((gigEl, index) => {
    // Get all inputs and textareas with data-field attribute
    const fields = gigEl.querySelectorAll('[data-field]');

    const gigData = {
      link: '',
      title: '',
      description: '',
      requirements_list: [],
      packages: {
        basic: { price: 0, quantity: '', deliveryTime: '', description: '' },
        standard: { price: 0, quantity: '', deliveryTime: '', description: '' },
        premium: { price: 0, quantity: '', deliveryTime: '', description: '' }
      }
    };

    fields.forEach(field => {
      const fieldName = field.getAttribute('data-field');
      const value = field.value.trim();

      switch (fieldName) {
        case 'link':
          gigData.link = value;
          break;
        case 'title':
          gigData.title = value;
          break;
        case 'description':
          gigData.description = value;
          break;
        case 'requirements':
          // Convert comma-separated string to array
          if (value) {
            gigData.requirements_list = value
              .split(',')
              .map(item => item.trim())
              .filter(item => item.length > 0);
          }
          break;
      }
    });

    // ✅ Collect package data
    const packageFields = gigEl.querySelectorAll('[data-package]');
    packageFields.forEach(field => {
      const packageTier = field.getAttribute('data-package'); // basic, standard, premium
      const pkgField = field.getAttribute('data-pkg-field'); // price, quantity, deliveryTime, description
      const value = field.value.trim();

      if (packageTier && pkgField && gigData.packages[packageTier]) {
        if (pkgField === 'price') {
          gigData.packages[packageTier][pkgField] = value ? parseFloat(value) : 0;
        } else {
          gigData.packages[packageTier][pkgField] = value;
        }
      }
    });

    // ✅ ENHANCED: Validation and debugging
    const hasMinimumData = gigData.link && gigData.title;
    const hasPackageData = (gigData.packages.basic && gigData.packages.basic.price > 0) ||
      (gigData.packages.standard && gigData.packages.standard.price > 0) ||
      (gigData.packages.premium && gigData.packages.premium.price > 0);
    const hasContent = gigData.description ||
      hasPackageData ||
      (gigData.requirements_list && gigData.requirements_list.length > 0);

    console.log(`\n🔍 Gig #${index + 1} Validation:`, {
      hasLink: !!gigData.link,
      hasTitle: !!gigData.title,
      hasMinimumData,
      hasPackageData,
      hasDescription: !!gigData.description,
      hasRequirements: gigData.requirements_list && gigData.requirements_list.length > 0,
      hasContent,
      willSave: hasMinimumData && hasContent
    });

    if (hasMinimumData && hasContent) {
      // ✅ PRIORITY: Check same position (Gig #X → Gig #X)
      // This is what user wants: Gig #1 compares with old Gig #1, Gig #2 with old Gig #2
      let oldGig = (state.gigs && state.gigs[index]) ? state.gigs[index] : null;

      // Fallback: If no gig at this position, try finding by link
      if (!oldGig && state.gigs) {
        oldGig = state.gigs.find(g => g.link === gigData.link);
        if (oldGig) {
          console.log(`   ℹ️ Gig #${index + 1}: Found by link match (gig moved position)`);
        }
      }

      // ✅ Compare ALL fields to detect changes
      let gigChanged = !oldGig;  // New gig = needs summary

      if (oldGig) {
        // Check if ANYTHING changed
        const linkChanged = oldGig.link !== gigData.link;
        const titleChanged = oldGig.title !== gigData.title;
        const descChanged = oldGig.description !== gigData.description;
        const reqChanged = JSON.stringify(oldGig.requirements_list || []) !== JSON.stringify(gigData.requirements_list || []);
        const pkgChanged = JSON.stringify(oldGig.packages || {}) !== JSON.stringify(gigData.packages || {});

        gigChanged = linkChanged || titleChanged || descChanged || reqChanged || pkgChanged;

        if (gigChanged) {
          console.log(`   🔄 Gig #${index + 1}: CHANGED - will generate NEW summary`);
          if (linkChanged) console.log(`      → Link changed`);
          if (titleChanged) console.log(`      → Title changed`);
          if (descChanged) console.log(`      → Description changed`);
          if (reqChanged) console.log(`      → Requirements changed`);
          if (pkgChanged) console.log(`      → Packages changed`);
        } else {
          console.log(`   ✅ Gig #${index + 1}: SAME - reusing existing summary (${oldGig.summary?.length || 0} chars)`);
        }
      } else {
        console.log(`   🆕 Gig #${index + 1}: NEW - will generate summary`);
      }

      if (gigChanged) {
        gigData._needsSummary = true;  // Generate new summary
      } else {
        gigData._needsSummary = false;  // Reuse existing summary
        gigData.summary = oldGig.summary;  // ✅ Keep old summary - NO AI call
      }

      gigs.push(gigData);
    } else {
      // Log why gig is being skipped
      if (!hasMinimumData) {
        console.log(`⏭️ Gig #${index + 1}: SKIPPED - Missing link or title`);
        console.log(`   Link: "${gigData.link || '(empty)'}"`);
        console.log(`   Title: "${gigData.title || '(empty)'}"`);
      } else if (!hasContent) {
        console.log(`⏭️ Gig #${index + 1}: SKIPPED - No content (packages, description, or requirements)`);
        console.log(`   Has description: ${!!gigData.description}`);
        console.log(`   Has packages: ${hasPackageData}`);
        console.log(`   Has requirements: ${gigData.requirements_list && gigData.requirements_list.length > 0}`);
      }
    }
  });

  console.log(`\n📊 Summary of gigs to save: ${gigs.length}`);
  gigs.forEach((gig, idx) => {
    console.log(`\nGig #${idx + 1} DATA TO SEND:`, {
      link: gig.link,
      title: gig.title,
      description: gig.description ? `${gig.description.substring(0, 50)}...` : '(empty)',
      hasPackages: !!(gig.packages && (gig.packages.basic.price > 0 || gig.packages.standard.price > 0 || gig.packages.premium.price > 0)),
      packages: gig.packages,
      requirements_list: gig.requirements_list,
      needsSummary: gig._needsSummary,
      hasSummary: !!gig.summary,
      summaryLength: gig.summary?.length || 0
    });
  });

  if (gigs.length === 0) {
    addLog('WARNING', 'No gigs to save. Please provide at least link and title.');
    showToast('Please provide at least Gig Link and Title', 'warning');
    return;
  }

  try {
    addLog('INFO', 'Saving gigs...');
    showToast('Saving gigs...', 'info');

    // ✅ NEW STRATEGY: Save raw data FIRST, generate summaries LATER
    console.log('\n' + '='.repeat(60));
    console.log('💾 STEP 1: SAVE RAW DATA (Fast - No AI calls)');
    console.log('='.repeat(60));
    console.log('Strategy: Save gig data immediately');
    console.log('  → User can close popup safely');
    console.log('  → Summaries will generate in background');
    console.log('='.repeat(60) + '\n');

    // Mark which gigs need summaries for later
    const gigsNeedingSummary = [];

    gigs.forEach((gig, index) => {
      const gigNumber = index + 1;

      // Keep track of which gigs need new summaries
      if (gig._needsSummary) {
        gigsNeedingSummary.push(gigNumber);
        console.log(`📝 Gig #${gigNumber}: Will generate summary after save`);
        // Remove the summary temporarily - will generate after save
        delete gig.summary;
      } else {
        console.log(`♻️ Gig #${gigNumber}: Has existing summary (${gig.summary?.length || 0} chars)`);
      }

      // Clean up flag before sending
      delete gig._needsSummary;
    });

    console.log(`\n📊 Save Plan:`);
    console.log(`   → ${gigs.length} gig(s) to save NOW`);
    console.log(`   → ${gigsNeedingSummary.length} gig(s) need summaries AFTER save`);
    console.log(`   → ${gigs.length - gigsNeedingSummary.length} gig(s) with existing summaries\n`);

    // Log gigs data before sending
    console.log('📦 Gigs to save (raw data):');
    gigs.forEach((gig, index) => {
      console.log(`\nGig #${index + 1}:`);
      console.log('  - Link:', gig.link);
      console.log('  - Title:', gig.title);
      console.log('  - Description:', gig.description?.substring(0, 50) + '...');
      console.log('  - Packages:', gig.packages);
      console.log('  - Requirements:', gig.requirements_list);
      console.log('  - Summary:', gig.summary ? `${gig.summary.substring(0, 80)}...` : 'Will generate after save');
    });

    console.log(`\n📤 Sending ${gigs.length} gig(s) to backend for IMMEDIATE save...`);
    const result = await api.saveGigs(gigs);

    console.log(`✅ STEP 1 COMPLETE: All gigs saved to database!\n`);

    console.log(`\n${'='.repeat(60)}`);
    console.log(`📥 BACKEND RESPONSE RECEIVED`);
    console.log(`${'='.repeat(60)}`);
    console.log(`Success: ${result.success}`);
    console.log(`Count: ${result.count}`);
    console.log(`Gigs returned: ${result.gigs?.length || 0}`);
    console.log(`${'='.repeat(60)}\n`);

    if (result.success) {
      // Log what backend sent back
      if (result.gigs && result.gigs.length > 0) {
        console.log(`📋 Backend returned ${result.gigs.length} gig(s):`);
        result.gigs.forEach((gig, index) => {
          console.log(`\n   Gig #${index + 1} FROM DATABASE:`, {
            _id: gig._id,
            link: gig.link,
            title: gig.title,
            hasPackages: !!(gig.packages && (gig.packages.basic || gig.packages.standard || gig.packages.premium)),
            packagesData: gig.packages,
            hasSummary: !!gig.summary,
            summaryLength: gig.summary?.length || 0
          });
        });
      }

      console.log(`\n💾 Updating local state with backend data...`);
      const oldCount = state.gigs?.length || 0;
      state.gigs = result.gigs || gigs;  // Use backend response if available
      console.log(`   Old gigs count: ${oldCount}`);
      console.log(`   New gigs count: ${state.gigs.length}`);

      saveState();
      console.log(`✅ State saved to Chrome storage\n`);

      addLog('SUCCESS', `Saved ${result.count} gig(s) successfully`);
      showToast(`Saved ${result.count} gig(s) successfully!`, 'success');

      // ✅ Verify packages were saved correctly
      if (result.gigs && result.gigs.length > 0) {
        console.log(`\n🔍 VERIFICATION - Checking if all data saved correctly:`);
        console.log(`${'='.repeat(60)}`);

        result.gigs.forEach((gig, index) => {
          console.log(`\n   Gig #${index + 1}: "${gig.title}"`);

          const hasPackages = gig.packages && (
            (gig.packages.basic && gig.packages.basic.price > 0) ||
            (gig.packages.standard && gig.packages.standard.price > 0) ||
            (gig.packages.premium && gig.packages.premium.price > 0)
          );

          if (hasPackages) {
            console.log(`      ✅ Packages: Saved`);
            if (gig.packages.basic?.price > 0) {
              console.log(`         Basic: $${gig.packages.basic.price} for ${gig.packages.basic.quantity} in ${gig.packages.basic.deliveryTime}`);
            }
            if (gig.packages.standard?.price > 0) {
              console.log(`         Standard: $${gig.packages.standard.price} for ${gig.packages.standard.quantity} in ${gig.packages.standard.deliveryTime}`);
            }
            if (gig.packages.premium?.price > 0) {
              console.log(`         Premium: $${gig.packages.premium.price} for ${gig.packages.premium.quantity} in ${gig.packages.premium.deliveryTime}`);
            }
          } else {
            console.warn(`      ⚠️ Packages: NOT saved or empty`);
          }

          console.log(`      ${gig.summary ? '✅' : '⚠️'} Summary: ${gig.summary ? `${gig.summary.length} chars` : 'Not saved'}`);
          console.log(`      ${gig.requirements_list?.length > 0 ? '✅' : '⚠️'} Requirements: ${gig.requirements_list?.length || 0} items`);
        });

        console.log(`\n${'='.repeat(60)}\n`);
      }

      // ✅ Re-render gigs UI with fresh database data
      console.log(`🔄 Re-rendering gigs form with database data...`);
      renderGigs();
      console.log(`✅ UI updated - form now shows saved data from database\n`);

      // Send message to background script
      chrome.runtime.sendMessage({
        action: 'updateGigs',
        gigs: state.gigs
      }).catch(err => {
        console.warn('Background script not available:', err);
      });

      console.log(`✅ STEP 1 COMPLETE - Data saved!\n`);

      // ✅ STEP 2: Generate summaries for changed gigs (in background)
      if (gigsNeedingSummary.length > 0) {
        console.log('='.repeat(60));
        console.log(`📝 STEP 2: GENERATING SUMMARIES (Background)`);
        console.log('='.repeat(60));
        console.log(`${gigsNeedingSummary.length} gig(s) need new summaries`);
        console.log('Note: This happens in background - you can close popup now');
        console.log('='.repeat(60) + '\n');

        // Generate summaries asynchronously (don't wait)
        generateGigSummariesInBackground(result.gigs, gigsNeedingSummary).catch(err => {
          console.error('Background summary generation error:', err);
        });
      }

      console.log(`🎉 SAVE COMPLETE! ${gigsNeedingSummary.length > 0 ? 'Summaries generating in background...' : ''}\n`);
    } else {
      throw new Error(result.message || 'Failed to save gigs');
    }
  } catch (error) {
    console.error('Error saving gigs:', error);
    addLog('ERROR', error.message || 'Failed to save gigs');
    showToast(error.message || 'Failed to save gigs', 'error');
  }
}

// Switch tab
function switchTab(tabName) {
  // Update tab buttons
  document.querySelectorAll('.tab').forEach(tab => {
    if (tab.getAttribute('data-tab') === tabName) {
      tab.classList.add('active');
    } else {
      tab.classList.remove('active');
    }
  });

  // Update tab content
  document.querySelectorAll('.tab-content').forEach(content => {
    content.classList.remove('active');
  });

  const activeContent = document.getElementById(`tab${tabName.charAt(0).toUpperCase() + tabName.slice(1)}`);
  if (activeContent) {
    activeContent.classList.add('active');
  }

  // Special handling for gigs tab
  if (tabName === 'gigs') {
    renderGigs();
  }

  // Special handling for payment tab - load plans
  if (tabName === 'payment') {
    loadPlans();
  }
}

// Render gigs
function renderGigs() {
  const container = document.getElementById('gigsContainer');
  if (!container) return;

  const limits = getEffectivePlanLimits();
  const limit = limits.gigs;
  const limitText = limit === -1 ? 'Unlimited' : limit;

  const gigsAvailableEl = document.getElementById('gigsAvailable');
  if (gigsAvailableEl) gigsAvailableEl.textContent = limitText;

  container.innerHTML = '';

  // Show message if limit reached
  if (limit !== -1 && state.gigs.length >= limit) {
    const upgradeMsg = document.createElement('div');
    upgradeMsg.style.cssText = 'padding: 12px; background: #fef3c7; border: 1px solid #f59e0b; border-radius: 8px; margin-bottom: 16px; color: #92400e; font-size: 13px;';
    upgradeMsg.innerHTML = `
      <strong>⚠️ Gig Limit Reached!</strong><br>
      You've used all ${limit} gigs. <a href="#" onclick="document.querySelector('[data-tab=\\'plans\\']').click(); return false;" style="color: #ea580c; text-decoration: underline;">Upgrade your plan</a> to add more gigs.
    `;
    container.appendChild(upgradeMsg);
  }

  const displayLimit = limit === -1 ? Math.max(10, state.gigs.length || 1) : limit;

  for (let i = 0; i < displayLimit; i++) {
    const gig = state.gigs[i] || {
      link: '',
      title: '',
      description: '',
      deliverables: [],
      requirements_list: []
    };

    // Convert arrays to comma-separated strings for display
    const requirementsStr = (gig.requirements_list || []).join(', ');

    // Package data (ensure defaults)
    const packages = gig.packages || { basic: {}, standard: {}, premium: {} };

    const gigEl = document.createElement('div');
    gigEl.className = 'gig-item';
    gigEl.innerHTML = `
      <h4>Gig #${i + 1}</h4>
      <div class="gig-inputs">
        <input type="text" placeholder="Gig Link *" value="${escapeHtml(gig.link || '')}" class="input" data-field="link">
        <input type="text" placeholder="Gig Title *" value="${escapeHtml(gig.title || '')}" class="input" data-field="title" maxlength="80">
        <textarea placeholder="Gig Description" class="input gig-textarea" data-field="description" rows="3" maxlength="1200">${escapeHtml(gig.description || '')}</textarea>
        
        <div class="gig-packages-section">
          <h5 style="margin: 10px 0 5px 0; font-size: 13px; color: white;">💰 Package Pricing</h5>
          
          <div style="margin-bottom: 15px; padding: 10px; background: rgba(255,255,255,0.05); border-radius: 5px;">
            <h6 style="margin: 0 0 8px 0; font-size: 12px; color: #10b981;">Basic Package</h6>
            <div style="display: grid; grid-template-columns: 80px 1fr 100px; gap: 8px; margin-bottom: 8px;">
              <input type="number" placeholder="Price ($)" value="${escapeHtml(packages.basic.price || '')}" class="input" data-package="basic" data-pkg-field="price" style="font-size: 12px;">
              <input type="text" placeholder="Quantity (e.g., 50 videos)" value="${escapeHtml(packages.basic.quantity || '')}" class="input" data-package="basic" data-pkg-field="quantity" style="font-size: 12px;">
              <input type="text" placeholder="Delivery" value="${escapeHtml(packages.basic.deliveryTime || '')}" class="input" data-package="basic" data-pkg-field="deliveryTime" style="font-size: 12px;">
            </div>
            <textarea placeholder="Features (comma separated): HD footage, Music, Captions, etc." class="input gig-textarea" data-package="basic" data-pkg-field="description" rows="2" style="font-size: 12px;" maxlength="30">${escapeHtml(packages.basic.description || '')}</textarea>
          </div>
          
          <div style="margin-bottom: 15px; padding: 10px; background: rgba(255,255,255,0.05); border-radius: 5px;">
            <h6 style="margin: 0 0 8px 0; font-size: 12px; color: #3b82f6;">Standard Package</h6>
            <div style="display: grid; grid-template-columns: 80px 1fr 100px; gap: 8px; margin-bottom: 8px;">
              <input type="number" placeholder="Price ($)" value="${escapeHtml(packages.standard.price || '')}" class="input" data-package="standard" data-pkg-field="price" style="font-size: 12px;">
              <input type="text" placeholder="Quantity (e.g., 100 videos)" value="${escapeHtml(packages.standard.quantity || '')}" class="input" data-package="standard" data-pkg-field="quantity" style="font-size: 12px;">
              <input type="text" placeholder="Delivery" value="${escapeHtml(packages.standard.deliveryTime || '')}" class="input" data-package="standard" data-pkg-field="deliveryTime" style="font-size: 12px;">
            </div>
            <textarea placeholder="Features (comma separated): All basic + Extra features" class="input gig-textarea" data-package="standard" data-pkg-field="description" rows="2" style="font-size: 12px;" maxlength="30">${escapeHtml(packages.standard.description || '')}</textarea>
          </div>
          
          <div style="margin-bottom: 10px; padding: 10px; background: rgba(255,255,255,0.05); border-radius: 5px;">
            <h6 style="margin: 0 0 8px 0; font-size: 12px; color: #f59e0b;">Premium Package</h6>
            <div style="display: grid; grid-template-columns: 80px 1fr 100px; gap: 8px; margin-bottom: 8px;">
              <input type="number" placeholder="Price ($)" value="${escapeHtml(packages.premium.price || '')}" class="input" data-package="premium" data-pkg-field="price" style="font-size: 12px;">
              <input type="text" placeholder="Quantity (e.g., 150 videos)" value="${escapeHtml(packages.premium.quantity || '')}" class="input" data-package="premium" data-pkg-field="quantity" style="font-size: 12px;">
              <input type="text" placeholder="Delivery" value="${escapeHtml(packages.premium.deliveryTime || '')}" class="input" data-package="premium" data-pkg-field="deliveryTime" style="font-size: 12px;">
            </div>
            <textarea placeholder="Features (comma separated): All standard + Premium features" class="input gig-textarea" data-package="premium" data-pkg-field="description" rows="2" style="font-size: 12px;" maxlength="30">${escapeHtml(packages.premium.description || '')}</textarea>
          </div>
        </div>
        
        <div class="gig-requirements-section">
          <h5 style="margin: 10px 0 5px 0; font-size: 13px; color: white;">📋 Requirements Questions</h5>
          <textarea placeholder="Questions for buyer (comma separated)&#10;Example: What is your brand name?, Do you have color preferences?" class="input gig-textarea" data-field="requirements" rows="3" maxlength="120">${escapeHtml(requirementsStr)}</textarea>
        </div>
      </div>
    `;

    container.appendChild(gigEl);
  }
}

// Add log entry
function addLog(type, message) {
  const now = new Date();
  const time = now.toTimeString().split(' ')[0];

  const log = { time, type, message };
  state.logs.unshift(log);

  // Keep only last 50 logs
  if (state.logs.length > 50) {
    state.logs = state.logs.slice(0, 50);
  }

  saveState();
  updateLogsUI();

  // Also create log in backend if logged in
  if (state.isLoggedIn && api) {
    try {
      api.createLog(type, message).catch(() => { });
    } catch (error) {
      // Ignore errors
    }
  }
}

// Update logs UI
function updateLogsUI() {
  const container = document.getElementById('logsContainer');
  if (!container) return;

  container.innerHTML = '';

  state.logs.forEach(log => {
    const logEl = document.createElement('div');
    logEl.className = 'log-entry';

    let typeClass = 'log-info';
    if (log.type === 'SUCCESS') typeClass = 'log-success';
    if (log.type === 'ERROR') typeClass = 'log-error';
    if (log.type === 'WARNING') typeClass = 'log-warning';

    logEl.innerHTML = `
      <span class="log-time">${escapeHtml(log.time)}</span>
      <span class="log-type ${typeClass}">${escapeHtml(log.type)}</span>
      <span class="log-message">${escapeHtml(log.message)}</span>
    `;

    container.appendChild(logEl);
  });
}

// Update username change status display
async function updateUsernameChangeStatus() {
  const infoDiv = document.getElementById('usernameChangeInfo');
  if (!infoDiv) return;

  try {
    const result = await api.getUsernameChangeStatus();

    if (result.success) {
      if (!result.allowed) {
        // Username is locked
        infoDiv.textContent = `🔒 ${result.reason}`;
        infoDiv.style.color = '#ef4444';
        infoDiv.style.display = 'block';
      } else if (result.changesRemaining !== undefined) {
        // Show remaining changes
        if (result.changesRemaining === 0) {
          infoDiv.textContent = '⚠️ No username changes remaining. Next change locks for 30 days.';
          infoDiv.style.color = '#fbbf24';
          infoDiv.style.display = 'block';
        } else if (result.changesRemaining === 1) {
          infoDiv.textContent = `ℹ️ ${result.changesRemaining} username change remaining in the next 30 days.`;
          infoDiv.style.color = '#ffd700';
          infoDiv.style.display = 'block';
        } else {
          infoDiv.textContent = `ℹ️ ${result.changesRemaining} username changes remaining.`;
          infoDiv.style.color = '#10b981';
          infoDiv.style.display = 'block';
        }
      } else {
        infoDiv.style.display = 'none';
      }
    } else {
      infoDiv.style.display = 'none';
    }
  } catch (error) {
    console.log('Could not fetch username change status:', error);
    infoDiv.style.display = 'none';
  }
}

// Handle refresh mode toggle
function handleRefreshModeToggle(event) {
  const isEnabled = event.target.checked;

  // ✨ FIX: Require existing interval to enable refresh mode
  if (isEnabled && (!state.timeInterval || state.timeInterval < 10)) {
    event.target.checked = false; // Revert toggle
    showToast('Interval Not Set', 'Please set a time interval first (minimum 10s)', 'error');
    return;
  }

  state.refreshMode = isEnabled;
  saveState();

  // Send message to background script
  chrome.runtime.sendMessage({
    action: 'setRefreshMode',
    enabled: isEnabled,
    interval: state.timeInterval // ✨ No fallback to 30
  }).catch(err => {
    console.warn('Background script not available:', err);
  });

  if (isEnabled) {
    showToast(
      'Refresh Mode Enabled ✅',
      `Bot will auto-refresh seller dashboard every ${state.timeInterval} seconds`,
      'success',
      4000
    );
    addLog('INFO', `✅ Refresh Mode enabled (interval: ${state.timeInterval}s)`);
  } else {
    showToast(
      'Refresh Mode Disabled',
      'Bot will work normally without auto-refresh',
      'info',
      3000
    );
    addLog('INFO', '⏸️ Refresh Mode disabled');
  }
}

// Update entire UI based on state
function updateUI() {
  // Always show dashboard (no separate login screen)
  const notLoggedInSection = document.getElementById('notLoggedIn');
  const loggedInSection = document.getElementById('loggedIn');

  if (state.isLoggedIn) {
    // Show user info, hide login button
    notLoggedInSection?.classList.add('hidden');
    loggedInSection?.classList.remove('hidden');

    // Update user info
    const emailEl = document.getElementById('userEmail');
    const planEl = document.getElementById('userPlan');

    if (emailEl) emailEl.textContent = state.userEmail;
    if (planEl) {
      planEl.textContent = state.userPlan;
      planEl.className = `badge badge-${state.userPlan.toLowerCase()}`;
    }

    // Update plan banner with remaining counts
    const planBanner = document.getElementById('planBanner');
    const planNameEl = document.getElementById('planName');
    const planLimitsEl = document.getElementById('planLimits');

    const effectiveLimits = getEffectivePlanLimits();
    const totalGigs = typeof effectiveLimits.gigs === 'number' ? effectiveLimits.gigs : -1;
    const totalReplies = typeof effectiveLimits.repliesPerMonth === 'number' ? effectiveLimits.repliesPerMonth : -1;
    const usedGigs = state.gigs?.length || 0;
    const usedReplies = state.repliesThisMonth || 0;
    const remainingGigs = totalGigs === -1 ? 'Unlimited' : Math.max(0, totalGigs - usedGigs);
    const remainingReplies = totalReplies === -1 ? 'Unlimited' : Math.max(0, totalReplies - usedReplies);

    // Show banner if not unlimited
    if (totalGigs === -1 && totalReplies === -1) {
      planBanner?.classList.add('hidden');
    } else {
      planBanner?.classList.remove('hidden');
      if (planNameEl) planNameEl.textContent = getPlanKey(state.userPlan);
      if (planLimitsEl) {
        const gigsText = totalGigs === -1 ? 'Unlimited gigs' : `${remainingGigs} gigs remaining`;
        const repliesText = totalReplies === -1 ? 'Unlimited replies' : `${remainingReplies} replies remaining`;
        planLimitsEl.textContent = `${gigsText}, ${repliesText}`;

        // Show warning if low
        const lowGigs = totalGigs !== -1 && remainingGigs !== 'Unlimited' && remainingGigs <= 1;
        const lowReplies = totalReplies !== -1 && remainingReplies !== 'Unlimited' && remainingReplies <= 50;
        if (lowGigs || lowReplies) {
          planLimitsEl.style.color = '#ef4444';
        } else {
          planLimitsEl.style.color = '';
        }
      }

      // ✨ NEW: Update plan expire date
      const planExpireEl = document.getElementById('planExpireText');
      if (planExpireEl && state.subscription?.endDate) {
        const endDate = new Date(state.subscription.endDate);
        const now = new Date();
        const daysLeft = Math.ceil((endDate - now) / (1000 * 60 * 60 * 24));

        const expireParent = planExpireEl.parentElement;

        if (daysLeft < 0) {
          // Expired
          planExpireEl.textContent = `Expired on ${endDate.toLocaleDateString()}`;
          expireParent.className = 'plan-expire expired';
        } else if (daysLeft <= 7) {
          // Expiring soon (warning)
          planExpireEl.textContent = `Expires in ${daysLeft} day${daysLeft !== 1 ? 's' : ''} (${endDate.toLocaleDateString()})`;
          expireParent.className = 'plan-expire warning';
        } else {
          // Normal
          planExpireEl.textContent = `Expires on ${endDate.toLocaleDateString()} (${daysLeft} days left)`;
          expireParent.className = 'plan-expire';
        }
      } else if (planExpireEl) {
        // No expire date or Lifetime plan
        planExpireEl.textContent = '';
        planExpireEl.parentElement.className = 'plan-expire';
      }
    }
  } else {
    // Show login button, hide user info
    notLoggedInSection?.classList.remove('hidden');
    loggedInSection?.classList.add('hidden');

    // Hide plan banner when not logged in
    document.getElementById('planBanner')?.classList.add('hidden');
  }

  // Update dashboard inputs (regardless of login state)
  if (state.fiverrUsername) {
    const usernameInput = document.getElementById('fiverrUsername');
    if (usernameInput) usernameInput.value = state.fiverrUsername;
  }
  if (state.timeInterval) {
    const intervalInput = document.getElementById('timeInterval');
    if (intervalInput) intervalInput.value = state.timeInterval;
  }

  // ✨ Update refresh mode toggle
  const refreshModeToggle = document.getElementById('refreshModeToggle');
  if (refreshModeToggle) {
    refreshModeToggle.checked = state.refreshMode || false;
  }

  // Update username change status if logged in
  if (state.isLoggedIn) {
    updateUsernameChangeStatus();
  }

  // Update bot status
  const botStatusEl = document.getElementById('botStatus');
  if (botStatusEl) botStatusEl.textContent = state.botStatus;

  // Update analytics
  const totalRepliesEl = document.getElementById('totalReplies');
  const avgTimeEl = document.getElementById('avgTime');
  const successRateEl = document.getElementById('successRate');
  const activeChatsEl = document.getElementById('activeChats');
  const activeConversationsEl = document.getElementById('activeConversations');
  const repliesTodayEl = document.getElementById('repliesToday');

  if (totalRepliesEl) totalRepliesEl.textContent = state.analytics.totalReplies;
  // Show "--" instead of "0s" when no replies have been sent
  if (avgTimeEl) {
    avgTimeEl.textContent = state.analytics.avgTime > 0
      ? state.analytics.avgTime + 's'
      : '--';
  }
  if (successRateEl) successRateEl.textContent = state.analytics.successRate + '%';
  if (activeChatsEl) activeChatsEl.textContent = state.analytics.activeChats;  // ✅ Lifetime unique (Analytics tab)
  if (activeConversationsEl) activeConversationsEl.textContent = state.analytics.activeConversations;  // ✅ Today's unique (Bot Controls)
  if (repliesTodayEl) repliesTodayEl.textContent = state.analytics.repliesToday; // ✅ Now shows today only!

  // Update logs
  updateLogsUI();
}

// Listen for messages from background script
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  try {
    if (message.action === 'updateAnalytics') {
      state.analytics = { ...state.analytics, ...message.data };
      saveState();
      updateUI();
      sendResponse({ success: true });
      return true;
    }

    if (message.action === 'updateRepliesUsage') {
      const { repliesThisMonth, repliesLimit } = message.data;
      state.repliesThisMonth = repliesThisMonth;
      const overrides = { ...(state.planLimits || {}) };
      if (typeof repliesLimit === 'number') {
        overrides.repliesPerMonth = repliesLimit;
      }
      state.planLimits = normalizePlanLimits(state.userPlan, overrides);

      saveState();
      updateUI();
      sendResponse({ success: true });
      return true;
    }

    if (message.action === 'addLog') {
      addLog(message.type, message.message);
      sendResponse({ success: true });
      return true;
    }

    if (message.action === 'botStatusChange') {
      state.botStatus = message.status;
      saveState();
      updateUI();
      sendResponse({ success: true });
      return true;
    }

    if (message.action === 'repliesLimitReached') {
      const { message: errorMessage, repliesThisMonth, repliesLimit } = message.data || {};
      state.repliesThisMonth = repliesThisMonth || state.repliesThisMonth;

      // Update bot status to offline
      state.botStatus = 'Offline';
      saveState();
      updateUI();

      // Show upgrade toast
      showToast(
        'Replies Limit Reached ⚠️',
        errorMessage || `You've used all ${repliesLimit || 'your'} replies this month. Please upgrade your plan to continue.`,
        'error',
        8000
      );

      // Switch to plans tab after a short delay
      setTimeout(() => {
        switchTab('payment');
        loadPlans();
      }, 2000);

      sendResponse({ success: true });
      return true;
    }

    if (message.action === 'planExpired') {
      const { message: errorMessage, expireDate } = message.data || {};

      // Update bot status to offline
      state.botStatus = 'Offline';
      saveState();
      updateUI();

      // Show upgrade toast
      const expireDateStr = expireDate ? new Date(expireDate).toLocaleDateString() : '';
      showToast(
        'Plan Expired ⚠️',
        errorMessage || `Your subscription expired${expireDateStr ? ' on ' + expireDateStr : ''}. Please upgrade your plan to continue using the bot.`,
        'error',
        8000
      );

      // Switch to plans tab after a short delay
      setTimeout(() => {
        switchTab('payment');
        loadPlans();
      }, 2000);

      sendResponse({ success: true });
      return true;
    }
  } catch (error) {
    console.error('Error handling message:', error);
    sendResponse({ success: false, error: error.message });
  }

  return false;
});
