// ===================================================================
// CONFIGURATION - Error Logging
// ===================================================================
const SHOW_ERRORS = false; // Set to true to show errors in console

// Override console.error and console.warn to suppress messages
if (!SHOW_ERRORS) {
  const originalError = console.error;
  const originalWarn = console.warn;

  console.error = function (...args) {
    // Suppress all error logs in production
    return;
  };

  console.warn = function (...args) {
    // Suppress all warning logs in production
    return;
  };
}

// ===================================================================
// GLOBAL ERROR HANDLERS - Prevent errors from showing in Chrome Extensions
// ===================================================================

// Catch unhandled promise rejections
self.addEventListener('unhandledrejection', (event) => {
  if (!SHOW_ERRORS) {
    event.preventDefault(); // Prevent the error from appearing in Chrome's error log
  }
});

// Catch runtime errors
self.addEventListener('error', (event) => {
  if (!SHOW_ERRORS) {
    event.preventDefault(); // Prevent the error from appearing in Chrome's error log
  }
});

// ===================================================================
// AUTH HELPERS
// ===================================================================

async function getAuthContext() {
  return new Promise((resolve) => {
    chrome.storage.local.get(['authToken', 'deviceId', 'backendUrl'], (data) => {
      resolve({
        token: data.authToken || null,
        deviceId: data.deviceId || null,
        backendUrl: data.backendUrl || 'https://fivreply-production.up.railway.app'
      });
    });
  });
}

async function buildAuthHeaders(includeJson = true) {
  const { token, deviceId } = await getAuthContext();
  const headers = {};
  if (includeJson) {
    headers['Content-Type'] = 'application/json';
  }
  if (token) {
    headers['Authorization'] = `Bearer ${token}`;
  }
  if (deviceId) {
    headers['X-Device-Id'] = deviceId;
  }
  return headers;
}
// Background service worker for Fiverr Auto Reply Bot

let botState = {
  isRunning: false,
  username: '',
  interval: null, // ✨ Changed to null - Must be set by user
  gigs: [],
  checkIntervalId: null,
  refreshMode: false,  // ✨ NEW: Refresh mode state
  refreshIntervalId: null,  // ✨ NEW: Refresh interval ID
  isAutomationRunning: false,  // ✨ NEW: Track if automation is currently executing
  refreshTimeoutId: null,  // ✨ NEW: Timeout ID for refresh after automation completes
  heartbeatIntervalId: null,  // ✨ NEW: Heartbeat interval to keep backend in sync
  stats: {
    totalReplies: 0,
    successCount: 0,
    errorCount: 0,
    activeConversations: 0
  }
};

// Initialize on installation
chrome.runtime.onInstalled.addListener(() => {
  console.log('✅ Fiverr Auto Reply Bot installed - VERSION 1.0.2 - UPDATED');
  console.log('🔧 Checking contextMenus API availability...');

  // Load saved state
  chrome.storage.local.get(['botState'], (result) => {
    if (result.botState) {
      botState = { ...botState, ...result.botState };
    }
  });

  // Create context menu (with error handling)
  try {
    if (chrome.contextMenus) {
      console.log('✅ contextMenus API is available');
      chrome.contextMenus.create({
        id: 'fiverrAutoReply',
        title: 'Fiverr Auto Reply Bot',
        contexts: ['page', 'action']
      });
      console.log('✅ Context menu created successfully');
    } else {
      console.warn('❌ contextMenus API not available - check manifest.json permissions');
    }
  } catch (error) {
    console.error('❌ Error creating context menu:', error);
    // Non-critical error, extension can still function
  }
});

// Save bot state
function saveBotState() {
  chrome.storage.local.set({ botState });
}

// Check if plan is expired or replies exhausted
async function checkPlanAndRepliesStatus() {
  try {
    const appStateData = await chrome.storage.local.get(['appState']);
    const appState = appStateData.appState || {};

    // Check if plan has expired
    if (appState.subscription && appState.subscription.endDate) {
      const endDate = new Date(appState.subscription.endDate);
      const now = new Date();

      if (endDate < now) {
        // Plan has expired
        sendLogToPopup('ERROR', '⚠️ Your subscription has expired.');
        sendLogToPopup('ERROR', '🛑 Bot stopped automatically. Please upgrade your plan to continue.');
        handleStopBot();

        // Notify popup to show upgrade message
        chrome.runtime.sendMessage({
          action: 'planExpired',
          data: {
            message: 'Your subscription has expired',
            expireDate: endDate.toISOString()
          }
        }).catch(() => { });

        return false;
      }
    }

    // Check if replies remaining is 0
    const repliesThisMonth = appState.repliesThisMonth || 0;
    const planLimits = appState.planLimits || {};
    const repliesLimit = typeof planLimits.repliesPerMonth === 'number' ? planLimits.repliesPerMonth : -1;

    // If not unlimited and reached limit
    if (repliesLimit !== -1 && repliesThisMonth >= repliesLimit) {
      sendLogToPopup('ERROR', '⚠️ Replies remaining limit reached.');
      sendLogToPopup('ERROR', '🛑 Bot stopped automatically. Please upgrade your plan to continue.');
      handleStopBot();

      // Notify popup to show upgrade message
      chrome.runtime.sendMessage({
        action: 'repliesLimitReached',
        data: {
          message: 'Reply limit reached',
          repliesThisMonth,
          repliesLimit
        }
      }).catch(() => { });

      return false;
    }

    return true;
  } catch (error) {
    console.error('Error checking plan status:', error);
    return true; // Don't stop bot on error
  }
}

// Listen for messages from popup and content scripts
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('Background received message:', message);

  switch (message.action) {
    case 'registerUsername':
      handleRegisterUsername(message.username);
      sendResponse({ success: true });
      break;

    case 'setInterval':
      handleSetInterval(message.interval);
      sendResponse({ success: true });
      break;

    case 'updateGigs':
      handleUpdateGigs(message.gigs);
      sendResponse({ success: true });
      break;

    case 'setRefreshMode':
      handleSetRefreshMode(message.enabled, message.interval);
      sendResponse({ success: true });
      break;

    case 'startBot':
      handleStartBot(message.gigs);
      sendResponse({ success: true });
      break;

    case 'stopBot':
      handleStopBot();
      sendResponse({ success: true });
      break;

    case 'newMessage':
      handleNewMessage(message.data);
      sendResponse({ success: true });
      break;

    case 'getState':
      sendResponse({ state: botState });
      break;

    case 'syncConversationToMongoDB':
      // Async MongoDB sync
      syncConversationToMongoDB(message.customerId, message.conversationData)
        .then(success => {
          if (success) {
            sendResponse({ success: true });
          } else {
            sendResponse({ success: false, error: 'MongoDB sync failed - check background console for details' });
          }
        })
        .catch(error => {
          console.error('Sync catch block error:', error);
          sendResponse({ success: false, error: error.message || error.toString() });
        });
      return true; // Keep channel open for async response

    case 'updateSummaryInMongoDB':
      // Async MongoDB summary update
      updateConversationSummaryInMongoDB(message.customerId, message.summary)
        .then(success => {
          if (success) {
            sendResponse({ success: true });
          } else {
            sendResponse({ success: false, error: 'Summary update failed - check background console' });
          }
        })
        .catch(error => {
          console.error('Summary update catch block error:', error);
          sendResponse({ success: false, error: error.message || error.toString() });
        });
      return true; // Keep channel open for async response

    case 'updateAnalyticsAfterReply':
      // Update analytics after successful reply
      updateAnalyticsAfterReply(message.success, message.responseTime, message.customerId)
        .then(success => {
          if (success) {
            sendResponse({ success: true });
          } else {
            sendResponse({ success: false, error: 'Analytics update failed' });
          }
        })
        .catch(error => {
          console.error('Analytics update error:', error);
          sendResponse({ success: false, error: error.message });
        });
      return true; // Keep channel open for async response

    case 'generateReplyFull':
      // ✅ NEW BACKEND-FIRST: Generate reply with full conversation management
      console.log('🚀 ===== BACKEND-FIRST REPLY REQUEST =====');
      console.log(`Customer ID: ${message.customerId}`);
      console.log(`Buyer Username: ${message.buyerUsername}`);
      // ✅ Seller name is fetched from database, not from extension
      console.log(`Message: "${message.newMessage?.substring(0, 50)}..."`);
      console.log(`Has Gig Info: ${!!message.gigInfo}`);

      handleGenerateReplyFull(message.customerId, message.newMessage, message.buyerUsername, message.gigInfo)
        .then(result => {
          console.log('✅ Backend reply success:', result.success);
          sendResponse(result);
        })
        .catch(error => {
          console.error('❌ Backend reply error:', error);
          sendResponse({
            success: false,
            error: error.message || 'Failed to generate reply'
          });
        });
      return true; // Keep channel open for async response

    case 'keepAlive':
      // Just receiving this keeps the SW alive
      // console.log('💓 Keep alive ping received');
      sendResponse({ success: true });
      break;

    default:
      // Don't send error for unknown actions - let other listeners handle them
      console.log('Unknown action (might be handled by other listeners):', message.action);
      // Return false to indicate we didn't handle this message
      return false;
  }

  // Keep message channel open for async response
  return true;
});

// Handle username registration
function handleRegisterUsername(username) {
  botState.username = username;
  saveBotState();

  sendLogToPopup('SUCCESS', `Username saved: ${username}`);
  console.log('Username saved:', username);
}

// Handle interval setting
function handleSetInterval(interval) {
  botState.interval = interval;
  saveBotState();

  // If bot is running, restart with new interval
  if (botState.isRunning) {
    handleStopBot();
    handleStartBot();
  }

  sendLogToPopup('SUCCESS', `Interval updated to ${interval} seconds`);
  console.log('Interval set to:', interval);
}

// Handle gigs update
function handleUpdateGigs(gigs) {
  botState.gigs = gigs;
  saveBotState();

  sendLogToPopup('SUCCESS', `Updated ${gigs.length} gig(s)`);
  console.log('Gigs updated:', gigs);
}

// ✨ NEW: Handle refresh mode toggle
function handleSetRefreshMode(enabled, interval) {
  console.log('🔄 Setting refresh mode:', enabled, 'interval:', interval);

  botState.refreshMode = enabled;
  botState.interval = interval || 30;
  saveBotState();

  // Clear existing refresh interval
  if (botState.refreshIntervalId) {
    clearInterval(botState.refreshIntervalId);
    botState.refreshIntervalId = null;
  }

  if (enabled) {
    // Start auto-refresh on seller dashboard
    console.log(`✅ Starting refresh mode with ${botState.interval}s interval`);

    // If bot is already running, start refresh mode immediately
    if (botState.isRunning) {
      console.log('🚀 Bot is running - starting refresh mode now');
      startSellerDashboardRefresh();
    } else {
      console.log('⚠️ Bot not running - refresh mode will activate when bot starts');
    }

    sendLogToPopup('SUCCESS', `Refresh mode enabled (${botState.interval}s interval)`);
  } else {
    console.log('⏸️ Refresh mode disabled');
    sendLogToPopup('INFO', 'Refresh mode disabled');
  }
}

// ✨ NEW: Start auto-refresh on seller dashboard
// This restarts the automation flow AFTER automation completes, not during execution
function startSellerDashboardRefresh() {
  // Clear any existing refresh timeout
  if (botState.refreshTimeoutId) {
    clearTimeout(botState.refreshTimeoutId);
    botState.refreshTimeoutId = null;
  }

  // Don't start timer if automation is currently running
  if (botState.isAutomationRunning) {
    console.log('⏸️ Automation is running - refresh will start after completion');
    return;
  }

  // Start countdown timer - will trigger AFTER automation completes
  console.log(`⏱️ Starting refresh countdown: ${botState.interval} seconds after automation completes`);
  scheduleNextRefresh();
}

// ✨ NEW: Schedule next refresh after interval
function scheduleNextRefresh() {
  // Clear any existing timeout
  if (botState.refreshTimeoutId) {
    clearTimeout(botState.refreshTimeoutId);
    botState.refreshTimeoutId = null;
  }

  // Check if bot is still running
  if (!botState.isRunning || !botState.refreshMode) {
    console.log('⚠️ Bot stopped or refresh mode disabled - not scheduling refresh');
    return;
  }

  // Check if automation is running
  if (botState.isAutomationRunning) {
    console.log('⏸️ Automation running - refresh scheduled after completion');
    return;
  }

  // Schedule refresh after interval
  botState.refreshTimeoutId = setTimeout(async () => {
    console.log('🔄 Restarting automation flow...');
    sendLogToPopup('INFO', '🔄 Restarting automation cycle...');

    // Check if bot is still running
    if (!botState.isRunning || !botState.refreshMode) {
      console.log('⚠️ Bot stopped or refresh disabled - cancelling refresh');
      return;
    }

    // Find any Fiverr tab (not just seller_dashboard)
    const tabs = await chrome.tabs.query({ url: '*://*.fiverr.com/*' });

    if (tabs && tabs.length > 0) {
      // Use the first Fiverr tab found
      const fiverrTab = tabs[0];
      console.log(`✅ Found Fiverr tab (ID: ${fiverrTab.id}) - ${fiverrTab.url}`);

      // Send message to content script to reset state
      chrome.tabs.sendMessage(fiverrTab.id, {
        action: 'restartAutomation',
        gigs: botState.gigs || []
      }, async (response) => {
        if (chrome.runtime.lastError) {
          console.log('⚠️ Could not send restart message:', chrome.runtime.lastError.message);
          // Content script might not be loaded, proceed anyway
          console.log('📍 Proceeding with automation restart...');
          sendLogToPopup('INFO', 'Starting automation...');

          // Wait a bit longer for tab to be ready
          await new Promise(resolve => setTimeout(resolve, 2000));

          // Start automation flow
          await startAutomationFlow(true);
        } else {
          console.log('✅ Restart signal sent to content script');

          // Wait longer for content script to process the reset
          await new Promise(resolve => setTimeout(resolve, 2000));

          // Now restart the full automation flow
          console.log('🚀 Triggering full automation flow...');
          sendLogToPopup('SUCCESS', '🔄 Bot automation restarted');
          await startAutomationFlow(true); // ✅ Use existing tab, don't create new
        }
      });
    } else {
      console.log('⚠️ No Fiverr tabs open - skipping automation restart');
      sendLogToPopup('WARNING', 'Please keep a Fiverr tab open (seller dashboard or inbox)');
    }
  }, botState.interval * 1000);

  console.log(`✅ Refresh scheduled: Will restart after ${botState.interval} seconds (when automation completes)`);
}

// Handle start bot
async function handleStartBot(gigs = []) {
  if (botState.isRunning) {
    sendLogToPopup('WARNING', 'Bot is already running');
    return;
  }

  if (!botState.username) {
    sendLogToPopup('ERROR', 'Please register username first');
    return;
  }

  // ✨ NEW: Ensure interval is set
  if (!botState.interval || botState.interval < 10) {
    sendLogToPopup('ERROR', 'Please set a valid Time Interval (minimum 10s)');
    return;
  }

  // Store gigs in botState
  botState.gigs = gigs || [];
  botState.isRunning = true;
  saveBotState();

  console.log(`✅ Bot received ${botState.gigs.length} gig(s) from popup`);

  sendLogToPopup('SUCCESS', '🚀 Bot started');
  sendLogToPopup('INFO', `📦 Loaded ${botState.gigs.length} gig(s) for matching`);
  sendStatusToPopup('Online');

  // Show notification
  chrome.notifications.create({
    type: 'basic',
    iconUrl: 'icons/icon48.png',
    title: 'Fiverr Auto Reply Bot',
    message: 'Bot is now running',
    priority: 2
  });

  console.log('Bot started - Starting automation flow...');

  // ✨ Start heartbeat to keep backend in sync (every 2 minutes)
  startHeartbeat();

  // ✨ Create offscreen document to keep SW alive
  setupOffscreenDocument('offscreen.html');

  // ✨ Check if refresh mode is enabled
  if (botState.refreshMode) {
    console.log('🔄 Refresh mode is enabled - starting auto-restart cycle');
    sendLogToPopup('INFO', '🔄 Refresh mode active - bot will auto-restart');
    startSellerDashboardRefresh();
  }

  // Start the complete automation flow
  await startAutomationFlow();
}

// Handle stop bot
function handleStopBot() {
  if (!botState.isRunning) {
    sendLogToPopup('WARNING', 'Bot is not running');
    return;
  }

  botState.isRunning = false;
  botState.isAutomationRunning = false; // ✨ Clear automation running flag
  saveBotState();

  // Stop checking for messages
  stopMessageChecking();

  // ✨ Stop heartbeat
  stopHeartbeat();

  // ✨ Close offscreen document
  closeOffscreenDocument();

  // ✨ Clear refresh timeout if active
  if (botState.refreshTimeoutId) {
    clearTimeout(botState.refreshTimeoutId);
    botState.refreshTimeoutId = null;
    console.log('🔄 Refresh timeout cleared');
  }

  // ✨ Clear refresh interval if active (legacy)
  if (botState.refreshIntervalId) {
    clearInterval(botState.refreshIntervalId);
    botState.refreshIntervalId = null;
    console.log('🔄 Refresh interval cleared');
  }

  sendLogToPopup('INFO', 'Bot stopped');
  sendStatusToPopup('Offline');

  console.log('Bot stopped');
}

// Start periodic message checking
function startMessageChecking() {
  // Clear any existing interval
  if (botState.checkIntervalId) {
    clearInterval(botState.checkIntervalId);
  }

  // Set up new interval
  botState.checkIntervalId = setInterval(() => {
    checkForNewMessages();
  }, botState.interval * 1000);

  // Check immediately on start
  checkForNewMessages();
}

// ✨ Start heartbeat to keep backend database in sync
function startHeartbeat() {
  // Clear any existing heartbeat
  if (botState.heartbeatIntervalId) {
    clearInterval(botState.heartbeatIntervalId);
  }

  // Send heartbeat every 2 minutes (120000ms) to confirm bot is still running
  botState.heartbeatIntervalId = setInterval(async () => {
    try {
      // Check plan and replies status during heartbeat
      const canContinue = await checkPlanAndRepliesStatus();
      if (!canContinue) {
        return; // Bot has been stopped, heartbeat will be cleared
      }

      const result = await chrome.storage.local.get(['authToken', 'backendUrl']);
      if (result.authToken) {
        const backendUrl = result.backendUrl || 'https://fivreply-production.up.railway.app';

        const response = await fetch(`${backendUrl}/api/user/bot-status`, {
          method: 'PUT',
          headers: {
            'Authorization': `Bearer ${result.authToken}`,
            'Content-Type': 'application/json'
          },
          body: JSON.stringify({ status: 'Online' })
        });

        if (response.ok) {
          console.log('💓 Heartbeat sent - Backend status confirmed: Online');
        } else {
          console.log('⚠️ Heartbeat failed - Backend may be offline');
        }
      }
    } catch (error) {
      console.log('⚠️ Heartbeat error:', error.message);
    }
  }, 120000); // 2 minutes

  console.log('💓 Heartbeat started - Will sync status every 2 minutes');
}

// ✨ Stop heartbeat
function stopHeartbeat() {
  if (botState.heartbeatIntervalId) {
    clearInterval(botState.heartbeatIntervalId);
    botState.heartbeatIntervalId = null;
    console.log('💓 Heartbeat stopped');
  }
}

// Stop message checking
function stopMessageChecking() {
  if (botState.checkIntervalId) {
    clearInterval(botState.checkIntervalId);
    botState.checkIntervalId = null;
  }
}

// Check for new messages on Fiverr
async function checkForNewMessages() {
  if (!botState.isRunning) return;

  // Check plan and replies status before processing
  const canContinue = await checkPlanAndRepliesStatus();
  if (!canContinue) {
    return; // Bot has been stopped due to plan expiration or replies limit
  }

  console.log('Checking for new messages...');

  try {
    // Query all Fiverr tabs
    const tabs = await chrome.tabs.query({ url: 'https://www.fiverr.com/*' });

    if (tabs.length === 0) {
      console.log('No Fiverr tabs open');
      return;
    }

    // Send message to content script in each tab
    for (const tab of tabs) {
      try {
        const response = await chrome.tabs.sendMessage(tab.id, {
          action: 'checkMessages'
        });

        if (response && response.newMessages > 0) {
          sendLogToPopup('INFO', `Found ${response.newMessages} new message(s)`);
        }
      } catch (error) {
        console.log('Error communicating with tab:', error);
      }
    }
  } catch (error) {
    console.error('Error checking messages:', error);
    sendLogToPopup('ERROR', 'Failed to check messages');
  }
}

// Handle new message detected by content script
async function handleNewMessage(data) {
  if (!botState.isRunning) return;

  console.log('New message detected:', data);

  // Generate AI response
  const reply = await generateAIReply(data.message, data.context);

  // Send reply through content script
  try {
    const tabs = await chrome.tabs.query({
      url: 'https://www.fiverr.com/*',
      active: true
    });

    if (tabs.length > 0) {
      await chrome.tabs.sendMessage(tabs[0].id, {
        action: 'sendReply',
        conversationId: data.conversationId,
        reply: reply
      });

      // Update stats
      botState.stats.totalReplies++;
      botState.stats.successCount++;
      saveBotState();

      // Update UI
      updateAnalytics();

      sendLogToPopup('SUCCESS', 'Reply sent successfully');

      // Show notification
      chrome.notifications.create({
        type: 'basic',
        iconUrl: 'icons/icon48.png',
        title: 'Reply Sent',
        message: `Auto-replied to message`,
        priority: 1
      });
    }
  } catch (error) {
    console.error('Error sending reply:', error);
    botState.stats.errorCount++;
    sendLogToPopup('ERROR', 'Failed to send reply');
  }
}

// Generate AI response (placeholder - can be enhanced with actual AI API)
async function generateAIReply(message, context) {
  // Simple rule-based responses for now
  // In production, this would call an AI API like OpenAI GPT

  const lowerMessage = message.toLowerCase();

  // Greeting responses
  if (lowerMessage.includes('hello') || lowerMessage.includes('hi')) {
    return "Hello! Thank you for your message. I'm here to help you. How can I assist you today?";
  }

  // Price inquiry
  if (lowerMessage.includes('price') || lowerMessage.includes('cost') || lowerMessage.includes('how much')) {
    return "Thank you for your interest! Please check my gig details for pricing information. I offer competitive rates and can discuss custom packages based on your needs.";
  }

  // Delivery time inquiry
  if (lowerMessage.includes('deliver') || lowerMessage.includes('how long') || lowerMessage.includes('time')) {
    return "I typically deliver within the timeframe specified in the gig. For urgent requests, I can offer express delivery. Please let me know your deadline and I'll do my best to accommodate.";
  }

  // Custom order inquiry
  if (lowerMessage.includes('custom') || lowerMessage.includes('requirements')) {
    return "I'd be happy to work on a custom order for you! Please share your specific requirements and I'll send you a custom offer tailored to your needs.";
  }

  // Sample/portfolio request
  if (lowerMessage.includes('sample') || lowerMessage.includes('portfolio') || lowerMessage.includes('example')) {
    return "Please check my gig portfolio for samples of my work. If you need specific examples related to your project, feel free to let me know and I can share relevant samples.";
  }

  // Default response
  return "Thank you for your message! I've received your inquiry and will get back to you shortly with more details. Is there anything specific you'd like to know about my services?";
}

// Update analytics in popup
function updateAnalytics() {
  const successRate = botState.stats.totalReplies > 0
    ? Math.round((botState.stats.successCount / botState.stats.totalReplies) * 100)
    : 0;

  const avgTime = botState.interval / 2; // Rough estimate

  chrome.runtime.sendMessage({
    action: 'updateAnalytics',
    data: {
      totalReplies: botState.stats.totalReplies,
      avgTime: avgTime,
      successRate: successRate,
      activeChats: botState.stats.activeConversations
    }
  }).catch(() => {
    // Popup might be closed, ignore error
  });
}

// Send log message to popup
function sendLogToPopup(type, message) {
  chrome.runtime.sendMessage({
    action: 'addLog',
    type: type,
    message: message
  }).catch(() => {
    // Popup might be closed, ignore error
  });
}

// Send status update to popup
function sendStatusToPopup(status) {
  chrome.runtime.sendMessage({
    action: 'botStatusChange',
    status: status
  }).catch(() => {
    // Popup might be closed, ignore error
  });
}

// Handle alarm events (for scheduled tasks)
chrome.alarms.onAlarm.addListener((alarm) => {
  if (alarm.name === 'checkMessages') {
    checkForNewMessages();
  }
});

// Keep service worker alive
chrome.runtime.onStartup.addListener(async () => {
  console.log('🔄 Service worker started - Browser/Extension restarted');

  // Restore bot state
  chrome.storage.local.get(['botState', 'authToken'], async (result) => {
    if (result.botState) {
      botState = { ...botState, ...result.botState };

      // If bot was running before restart, force it to Offline
      // This ensures database is in sync with actual bot state
      if (botState.isRunning) {
        console.log('⚠️ Bot was running before restart - Resetting to Offline');
        console.log('💡 User must manually start bot again');

        // Force stop the bot
        botState.isRunning = false;
        botState.isAutomationRunning = false;
        saveBotState();

        // Update backend database to Offline
        if (result.authToken) {
          try {
            const backendUrl = await new Promise((resolve) => {
              chrome.storage.local.get(['backendUrl'], (data) => {
                resolve(data.backendUrl || 'https://fivreply-production.up.railway.app');
              });
            });

            const response = await fetch(`${backendUrl}/api/user/bot-status`, {
              method: 'PUT',
              headers: {
                'Authorization': `Bearer ${result.authToken}`,
                'Content-Type': 'application/json'
              },
              body: JSON.stringify({ status: 'Offline' })
            });

            if (response.ok) {
              console.log('✅ Backend database updated: botStatus = Offline');
            } else {
              console.log('⚠️ Failed to update backend database (will retry on next action)');
            }
          } catch (error) {
            console.log('⚠️ Could not reach backend (offline or not running):', error.message);
          }
        }

        // Notify popup about status change
        sendStatusToPopup('Offline');
        sendLogToPopup('INFO', '⚠️ Bot was stopped due to browser restart. Please start manually.');
      }
    }
  });
});

// Handle extension suspend (when service worker is being terminated)
chrome.runtime.onSuspend.addListener(() => {
  console.log('⚠️ Service worker suspending - Cleaning up bot state');

  // Stop heartbeat if running
  stopHeartbeat();

  // If bot is running, mark it for cleanup on next startup
  if (botState.isRunning) {
    console.log('⚠️ Bot is running - will be stopped on next startup');
    // The onStartup handler will reset the database status
  }
});

// ============================================================================
// COMPLETE AUTOMATION FLOW WITH LOOP
// ============================================================================

/**
 * Main automation flow - loops through all unread messages
 * @param {boolean} useExistingTab - If true, use any existing Fiverr tab instead of creating new one
 */
async function startAutomationFlow(useExistingTab = false) {
  if (!botState.isRunning) {
    console.log('Bot stopped, exiting flow');
    return;
  }

  // Check plan and replies status before starting automation
  const canContinue = await checkPlanAndRepliesStatus();
  if (!canContinue) {
    return; // Bot has been stopped due to plan expiration or replies limit
  }

  // ✨ Mark automation as running (prevent refresh during execution)
  botState.isAutomationRunning = true;
  console.log('🚀 Automation started - refresh paused until completion');

  // Clear any pending refresh timeout (automation is running now)
  if (botState.refreshTimeoutId) {
    clearTimeout(botState.refreshTimeoutId);
    botState.refreshTimeoutId = null;
    console.log('⏸️ Cleared pending refresh - automation is running');
  }

  try {
    sendLogToPopup('INFO', '📱 Checking for existing Fiverr tabs...');

    let dashboardTab;

    if (useExistingTab) {
      // For refresh mode: Use ANY existing Fiverr tab (don't create new)
      console.log('🔄 Refresh mode: Looking for any existing Fiverr tab...');
      let existingTabs = await chrome.tabs.query({
        url: '*://*.fiverr.com/*'
      });

      if (existingTabs.length > 0) {
        dashboardTab = existingTabs[0];
        console.log(`✅ Using existing Fiverr tab (ID: ${dashboardTab.id}) - ${dashboardTab.url}`);
        sendLogToPopup('SUCCESS', `✅ Using existing tab (ID: ${dashboardTab.id})`);

        // Navigate to seller dashboard if not already there
        if (!dashboardTab.url.includes('seller_dashboard')) {
          console.log('📍 Navigating to seller dashboard...');
          await chrome.tabs.update(dashboardTab.id, {
            url: 'https://www.fiverr.com/seller_dashboard',
            active: true
          });
          console.log('⏳ Waiting for navigation to complete...');
          await new Promise(resolve => setTimeout(resolve, 6000)); // Longer wait for navigation + page load
        } else {
          // Already on dashboard, just focus the tab
          console.log('✅ Already on seller dashboard');
          await chrome.windows.update(dashboardTab.windowId, { focused: true });
          await chrome.tabs.update(dashboardTab.id, { active: true });
          await new Promise(resolve => setTimeout(resolve, 2000));
        }
      } else {
        console.log('⚠️ No Fiverr tabs found - cannot restart automation');
        sendLogToPopup('WARNING', 'No Fiverr tabs open. Please open seller dashboard.');
        return;
      }
    } else {
      // Normal mode: Check for seller_dashboard tab, create if needed
      // Step 1: Check for existing dashboard tab first (across all windows)
      let existingTabs = await chrome.tabs.query({
        url: 'https://www.fiverr.com/seller_dashboard*'
      });

      if (existingTabs.length > 0) {
        // Use existing tab
        dashboardTab = existingTabs[0];
        sendLogToPopup('SUCCESS', `✅ Using existing dashboard tab (ID: ${dashboardTab.id})`);

        // Make sure the tab's window is focused and tab is active
        await chrome.windows.update(dashboardTab.windowId, { focused: true });
        await chrome.tabs.update(dashboardTab.id, { active: true });

        await new Promise(resolve => setTimeout(resolve, 2000)); // Short wait
      } else {
        // Create new tab only if none exists
        sendLogToPopup('INFO', '📱 No existing dashboard tab found. Opening new...');
        dashboardTab = await chrome.tabs.create({
          url: 'https://www.fiverr.com/seller_dashboard',
          active: true
        });
        await new Promise(resolve => setTimeout(resolve, 5000)); // Longer wait for new tab
      }
    }

    // Step 2: Verify username
    sendLogToPopup('INFO', '🔍 Verifying Fiverr username...');

    let verifyResult;
    try {
      verifyResult = await chrome.tabs.sendMessage(dashboardTab.id, {
        action: 'verifyUsername',
        expectedUsername: botState.username
      });
    } catch (error) {
      sendLogToPopup('ERROR', `❌ Failed to communicate with dashboard: ${error.message}`);
      sendLogToPopup('INFO', '🔄 Retrying in next cycle...');
      restartAfterInterval();
      return;
    }


    if (!verifyResult || !verifyResult.verified) {
      sendLogToPopup('ERROR', `❌ Username verification failed! Expected: ${botState.username}, Found: ${verifyResult?.foundUsername || 'none'}`);
      sendLogToPopup('ERROR', '⚠️ Bot stopped - Wrong account!');
      botState.isRunning = false;
      saveBotState();
      sendStatusToPopup('Offline');
      return;
    }

    sendLogToPopup('SUCCESS', `✅ Username verified: ${verifyResult.foundUsername}`);

    // ✨ NEW: Update seller name in backend if found
    if (verifyResult.sellerName) {
      console.log(`👤 Found seller name: "${verifyResult.sellerName}" - Updating backend...`);
      try {
        const { token: authToken, backendUrl } = await getAuthContext();
        if (authToken) {
          fetch(`${backendUrl}/api/user/seller-name`, {
            method: 'PUT',
            headers: {
              'Authorization': `Bearer ${authToken}`,
              'Content-Type': 'application/json'
            },
            body: JSON.stringify({ sellerName: verifyResult.sellerName })
          }).then(res => {
            if (res.ok) console.log('✅ Seller name updated in backend');
            else console.warn('⚠️ Failed to update seller name');
          }).catch(err => console.error('Error updating seller name:', err));
        }
      } catch (err) {
        console.error('Error getting auth context for seller name update:', err);
      }
    }

    // Step 3: Navigate to inbox
    sendLogToPopup('INFO', '📬 Navigating to inbox...');

    let inboxResult;
    try {
      inboxResult = await chrome.tabs.sendMessage(dashboardTab.id, {
        action: 'clickInboxLink'
      });
    } catch (error) {
      sendLogToPopup('ERROR', `❌ Failed to navigate to inbox: ${error.message}`);
      sendLogToPopup('INFO', '🔄 Retrying in next cycle...');
      restartAfterInterval();
      return;
    }

    if (!inboxResult || !inboxResult.success) {
      sendLogToPopup('ERROR', '❌ Failed to navigate to inbox');
      restartAfterInterval();
      return;
    }

    sendLogToPopup('SUCCESS', '✅ Inbox page opened');

    // Wait for inbox page to load
    await new Promise(resolve => setTimeout(resolve, 7000));

    // Find inbox tab (URL changed after navigation)
    const inboxTabs = await chrome.tabs.query({
      url: 'https://www.fiverr.com/inbox*',
      currentWindow: true
    });

    if (inboxTabs.length === 0) {
      sendLogToPopup('ERROR', '❌ Inbox tab not found');
      restartAfterInterval();
      return;
    }

    const inboxTabId = inboxTabs[0].id;
    sendLogToPopup('INFO', `📋 Inbox tab found (ID: ${inboxTabId})`);

    // Step 4: Process all unread messages in a loop
    await processAllUnreadMessages(inboxTabId);

  } catch (error) {
    console.error('Error in automation flow:', error);
    sendLogToPopup('ERROR', `❌ Automation error: ${error.message}`);
    restartAfterInterval();
  } finally {
    // ✨ Mark automation as complete
    botState.isAutomationRunning = false;
    console.log('✅ Automation completed - refresh can now be scheduled');

    // ✨ If refresh mode is enabled, schedule next refresh after interval
    if (botState.refreshMode && botState.isRunning) {
      console.log(`⏱️ Automation complete - scheduling refresh after ${botState.interval}s interval`);
      scheduleNextRefresh();
    }
  }
}

/**
 * Process all unread messages in a loop
 */
async function processAllUnreadMessages(inboxTabId) {
  let conversationCount = 0;

  while (botState.isRunning) {
    sendLogToPopup('INFO', `🔄 Checking for unread messages... (Processed: ${conversationCount})`);

    // Try to click an unread contact (with retry logic)
    const unreadResult = await clickUnreadWithRetry(inboxTabId);

    if (!unreadResult || !unreadResult.success) {
      // No more unread messages found - Try manual refresh sequence (if refresh mode is OFF)
      sendLogToPopup('INFO', '🔍 No unread messages found, checking manual refresh sequence...');

      try {
        const moreUnreadsResult = await chrome.tabs.sendMessage(inboxTabId, {
          action: 'checkMoreUnreads'
        });

        if (moreUnreadsResult && moreUnreadsResult.success && moreUnreadsResult.hasMore) {
          sendLogToPopup('SUCCESS', '✅ Found more unread messages after manual refresh! Continuing...');
          continue; // Continue the loop to process newly found unread messages
        }
      } catch (error) {
        console.error('Error during checkMoreUnreads:', error);
        sendLogToPopup('WARNING', '⚠️ Manual refresh check failed');
      }

      // No more unread messages (even after manual refresh attempt)
      sendLogToPopup('SUCCESS', `✅ All unread messages processed! Total: ${conversationCount}`);

      // ✅ Navigate back to dashboard instead of reload (fresh start from step 1)
      if (conversationCount > 0) {
        sendLogToPopup('INFO', '🔄 Going back to dashboard for fresh start...');
        try {
          await chrome.tabs.update(inboxTabId, {
            url: 'https://www.fiverr.com/seller_dashboard'
          });
          console.log('✅ Navigated back to dashboard');
          sendLogToPopup('SUCCESS', '✅ Back to dashboard - ready for next cycle');

          // Wait for dashboard to load
          await new Promise(resolve => setTimeout(resolve, 4000));
        } catch (error) {
          console.error('Error navigating to dashboard:', error);
          sendLogToPopup('WARNING', '⚠️ Could not navigate to dashboard');
        }
      } else {
        sendLogToPopup('INFO', 'ℹ️ No messages processed, staying on current page');
      }

      sendLogToPopup('INFO', `⏰ Waiting ${botState.interval} seconds before next check...`);

      // Show completion notification
      chrome.notifications.create({
        type: 'basic',
        iconUrl: 'icons/icon48.png',
        title: conversationCount > 0 ? '✅ All Messages Replied!' : 'ℹ️ No New Messages',
        message: conversationCount > 0
          ? `Processed ${conversationCount} conversation(s). Back to dashboard.`
          : 'No unread messages found. Waiting for next interval...',
        priority: 1
      });

      // Wait for time interval, then restart
      restartAfterInterval();
      return;
    }

    sendLogToPopup('SUCCESS', '✅ Unread contact clicked');

    // Wait for conversation page to load
    await new Promise(resolve => setTimeout(resolve, 5000));

    // Find conversation tab (should be active)
    const conversationTabs = await chrome.tabs.query({
      url: 'https://www.fiverr.com/*',
      currentWindow: true,
      active: true
    });

    if (conversationTabs.length === 0) {
      sendLogToPopup('ERROR', '❌ Conversation tab not found');
      continue;
    }

    const conversationTabId = conversationTabs[0].id;

    // Get user's gigs from botState
    const userGigs = botState.gigs || [];
    sendLogToPopup('INFO', `📦 Loaded ${userGigs.length} gig(s) from database`);

    // Log each gig for debugging
    if (userGigs.length > 0) {
      sendLogToPopup('INFO', '📋 User\'s Gigs Available for Matching:');
      userGigs.forEach((gig, index) => {
        const gigPath = gig.link.includes('fiverr.com') ? new URL(gig.link).pathname : gig.link;
        sendLogToPopup('INFO', `  Gig #${index + 1}: ${gig.title}`);
        sendLogToPopup('INFO', `     Link: ${gigPath}`);
      });
      sendLogToPopup('INFO', '');
    } else {
      sendLogToPopup('WARNING', '⚠️ No gigs found in database. Add gigs to enable gig-specific replies.');
    }

    // Process this conversation (read, reply, send with gig context)
    sendLogToPopup('INFO', '💬 Reading and replying to message...');

    let replyResult;
    try {
      replyResult = await chrome.tabs.sendMessage(conversationTabId, {
        action: 'readAndReply',
        apiKey: 'sk-6NgzImqoz-akTpiFLMeJsoGHEGDcgNGf6mOUWae9Y0T3BlbkFJB1Nnq3r3CWDdrTS3LAoFJGtdnM4tNrKTgH77tRIjgA',
        userGigs: userGigs  // Pass user's gigs
      });
    } catch (error) {
      sendLogToPopup('ERROR', `❌ Failed to process conversation: ${error.message}`);
      sendLogToPopup('INFO', '⏩ Skipping to next conversation...');
      botState.stats.errorCount++;
      saveBotState();

      // Navigate back to inbox and continue with next unread
      await chrome.tabs.update(inboxTabId, { url: 'https://www.fiverr.com/inbox' });
      await new Promise(resolve => setTimeout(resolve, 5000));
      continue;
    }

    if (replyResult && replyResult.success) {
      conversationCount++;

      // Show message truncation info if applicable
      if (replyResult.truncated) {
        sendLogToPopup('WARNING', `✂️ Long conversation: Using ${replyResult.messageCount}/${replyResult.totalMessages} most recent messages`);
      } else if (replyResult.messageCount && replyResult.totalMessages) {
        sendLogToPopup('INFO', `📊 Processed ${replyResult.totalMessages} message(s)`);
      }

      // Show gig matching result
      sendLogToPopup('INFO', '');
      sendLogToPopup('INFO', '🔍 Gig Matching Result:');

      if (replyResult.gigMatched) {
        const gigNum = replyResult.gigNumber || '?';
        sendLogToPopup('SUCCESS', `✅ MATCH FOUND! Gig #${gigNum}`);
        sendLogToPopup('SUCCESS', `   Title: "${replyResult.matchedGigTitle}"`);
        sendLogToPopup('SUCCESS', `   Link: ${replyResult.gigLink || 'N/A'}`);
        sendLogToPopup('INFO', '🎯 AI will use gig-specific context (title, description, packages, requirements)');
      } else if (replyResult.gigLink) {
        sendLogToPopup('WARNING', `❌ NO MATCH - Gig link found but not in database`);
        sendLogToPopup('WARNING', `   Extracted link: ${replyResult.gigLink}`);
        sendLogToPopup('WARNING', `   💡 Add this gig to database for specific replies`);
        sendLogToPopup('INFO', '💭 AI will use generic context only');
      } else {
        sendLogToPopup('INFO', '⚠️ No gig link detected in conversation page');
        sendLogToPopup('INFO', '   This may be a general inquiry or custom order discussion');
        sendLogToPopup('INFO', '💭 AI will use generic context only');
      }

      sendLogToPopup('INFO', '');

      sendLogToPopup('SUCCESS', `✅ Reply sent successfully! (${conversationCount} total)`);

      const replyPreview = replyResult.reply.length > 60
        ? replyResult.reply.substring(0, 60) + '...'
        : replyResult.reply;
      sendLogToPopup('INFO', `💬 Reply: "${replyPreview}"`);

      // Update stats
      botState.stats.totalReplies++;
      botState.stats.successCount++;
      saveBotState();
      updateAnalytics();
    } else {
      sendLogToPopup('ERROR', `❌ Failed to send reply: ${replyResult?.error || 'Unknown error'}`);
      botState.stats.errorCount++;
      saveBotState();
    }

    // Navigate back to inbox for next unread
    sendLogToPopup('INFO', '🔙 Navigating back to main inbox...');
    await chrome.tabs.update(inboxTabId, { url: 'https://www.fiverr.com/inbox' });

    // Wait for inbox to reload properly
    await new Promise(resolve => setTimeout(resolve, 5000));
  }
}

/**
 * Click unread contact with retry logic (3 attempts)
 */
async function clickUnreadWithRetry(tabId, maxRetries = 3) {
  for (let attempt = 1; attempt <= maxRetries; attempt++) {
    try {
      console.log(`Attempt ${attempt}/${maxRetries} to click unread...`);

      const result = await chrome.tabs.sendMessage(tabId, {
        action: 'clickUnreadContact'
      });

      if (result && result.success) {
        console.log(`✅ Unread contact clicked on attempt ${attempt}`);
        return result;
      }

      // If no unread found, return immediately (don't retry)
      if (result && result.error === 'No unread contacts found') {
        return result;
      }

      // Wait before retry
      if (attempt < maxRetries) {
        await new Promise(resolve => setTimeout(resolve, 2000));
      }
    } catch (error) {
      console.warn(`Attempt ${attempt} failed:`, error.message);
      if (attempt < maxRetries) {
        await new Promise(resolve => setTimeout(resolve, 2000));
      }
    }
  }

  return { success: false, error: 'All retry attempts failed' };
}

/**
 * Restart automation after time interval
 */
async function restartAfterInterval() {
  if (!botState.isRunning) {
    console.log('Bot stopped, not restarting');
    return;
  }

  // ✨ If refresh mode is enabled, use scheduleNextRefresh instead
  if (botState.refreshMode) {
    console.log('🔄 Refresh mode enabled - using scheduleNextRefresh');
    scheduleNextRefresh();
    return;
  }

  // Normal mode: Use setTimeout (old behavior)
  const intervalMs = botState.interval * 1000;
  const intervalMinutes = Math.round(botState.interval / 60 * 10) / 10;

  sendLogToPopup('INFO', `⏳ Restarting in ${intervalMinutes} minute(s)...`);

  setTimeout(async () => {
    if (!botState.isRunning) {
      console.log('Bot stopped during wait, not restarting');
      return;
    }

    sendLogToPopup('INFO', '🔄 Restarting automation flow...');

    // Check if inbox tab still exists
    const existingInboxTabs = await chrome.tabs.query({
      url: 'https://www.fiverr.com/inbox*'
    });

    if (existingInboxTabs.length > 0) {
      // Inbox tab exists, reuse it!
      const inboxTabId = existingInboxTabs[0].id;
      sendLogToPopup('SUCCESS', '✅ Reusing existing inbox tab');

      // Focus the inbox tab
      await chrome.windows.update(existingInboxTabs[0].windowId, { focused: true });
      await chrome.tabs.update(inboxTabId, { active: true });

      // Wait a bit for tab to be active
      await new Promise(resolve => setTimeout(resolve, 2000));

      // Directly process unread messages (skip dashboard & username check)
      await processAllUnreadMessages(inboxTabId);
    } else {
      // No inbox tab found, start full flow
      sendLogToPopup('INFO', '⚠️ Inbox tab not found, starting full flow...');
      startAutomationFlow();
    }
  }, intervalMs);
}

// ===================================================================
// MONGODB CONVERSATION SYNC
// ===================================================================

/**
 * Sync conversation to MongoDB backend
 * @param {string} customerId - Customer identifier
 * @param {Object} conversationData - Conversation data from Chrome storage
 * @returns {Promise<boolean>} Success status
 */
async function syncConversationToMongoDB(customerId, conversationData) {
  console.log('');
  console.log('=== 🔄 MONGODB SYNC START ===');
  console.log(`📋 Customer ID: ${customerId}`);
  console.log(`📊 Message count: ${conversationData.messages ? conversationData.messages.length : 0}`);
  console.log(`📝 Has summary: ${conversationData.summary ? 'Yes ✅' : 'No ❌'}`);
  console.log(`🎯 Has gig context: ${conversationData.gig_context ? 'Yes ✅' : 'No ❌'}`);

  try {
    // Get auth token (key is 'authToken' as saved by api.js)
    const tokenData = await chrome.storage.local.get(['authToken']);
    const authToken = tokenData.authToken;

    console.log(`🔑 Auth token: ${authToken ? 'Present ✅' : 'Missing ❌'}`);

    if (!authToken) {
      console.warn('⚠️ No auth token found, skipping MongoDB sync');
      sendLogToPopup('WARNING', '⚠️ Not logged in - conversation saved locally only');
      return false;
    }

    // Get backend URL from config
    const configData = await chrome.storage.local.get(['backendUrl']);
    const backendUrl = configData.backendUrl || 'https://fivreply-production.up.railway.app';

    console.log(`🌐 Backend URL: ${backendUrl}`);

    // Prepare conversation data for MongoDB
    const payload = {
      customer_id: customerId,
      summary: conversationData.summary || '',
      messages: conversationData.messages || [],
      gig_context: conversationData.gig_context || null,
      status: 'active'
    };

    console.log(`📤 Sending to: ${backendUrl}/api/conversations`);
    console.log(`📦 Payload size: ${JSON.stringify(payload).length} bytes`);

    const headers = await buildAuthHeaders(true);

    const response = await fetch(`${backendUrl}/api/conversations`, {
      method: 'POST',
      headers,
      body: JSON.stringify(payload)
    });

    console.log(`📥 Response status: ${response.status} ${response.statusText}`);

    if (!response.ok) {
      const errorData = await response.json().catch(() => ({ message: 'Unknown error' }));
      console.error('❌ Server returned error:', errorData);
      throw new Error(errorData.message || 'MongoDB sync failed');
    }

    const result = await response.json();
    console.log('✅ Conversation synced to MongoDB successfully');
    console.log(`💾 Saved conversation ID: ${result.conversation?._id || 'N/A'}`);
    console.log('=== ✅ MONGODB SYNC COMPLETE ===');
    console.log('');

    // Send log to popup
    sendLogToPopup('SUCCESS', `💾 Conversation saved to database (${customerId})`);

    return true;

  } catch (error) {
    console.error('=== ❌ MONGODB SYNC FAILED ===');
    console.error('Error type:', error.constructor.name);
    console.error('Error message:', error.message);
    console.error('Error details:', error);
    console.error('Stack trace:', error.stack);
    console.log('');

    // Don't fail the whole process if MongoDB sync fails
    // Local storage still has the data
    const errorMessage = error.message || error.toString() || 'Unknown error';
    sendLogToPopup('WARNING', `⚠️ Database sync failed: ${errorMessage}`);
    return false;
  }
}

/**
 * Update conversation summary in MongoDB
 * @param {string} customerId - Customer identifier
 * @param {string} summary - Summary text
 * @returns {Promise<boolean>} Success status
 */
async function updateConversationSummaryInMongoDB(customerId, summary) {
  console.log(`🔄 Updating summary in MongoDB for: ${customerId}`);

  try {
    const tokenData = await chrome.storage.local.get(['authToken']);
    const authToken = tokenData.authToken;

    if (!authToken) {
      console.warn('⚠️ No auth token, skipping summary sync');
      return false;
    }

    const configData = await chrome.storage.local.get(['backendUrl']);
    const backendUrl = configData.backendUrl || 'https://fivreply-production.up.railway.app';

    const headers = await buildAuthHeaders(true);

    const response = await fetch(`${backendUrl}/api/conversations/summary`, {
      method: 'POST',
      headers,
      body: JSON.stringify({
        customer_id: customerId,
        summary: summary
      })
    });

    if (!response.ok) {
      throw new Error('Failed to update summary in MongoDB');
    }

    console.log('✅ Summary updated in MongoDB');
    return true;

  } catch (error) {
    console.error('❌ MongoDB summary update error:', error);
    return false;
  }
}

/**
 * Update analytics after successful reply
 * @param {boolean} success - Whether reply was successful
 * @param {number} responseTime - Response time in seconds
 * @param {string} customerId - Customer ID for unique tracking
 * @returns {Promise<boolean>} Success status
 */
async function updateAnalyticsAfterReply(success = true, responseTime = 0, customerId = null) {
  console.log('📊 Updating analytics after reply...');
  console.log(`   Response time: ${responseTime}s, Success: ${success}, Customer: ${customerId}`);

  try {
    // Get auth token
    const { token: authToken, backendUrl, deviceId } = await getAuthContext();

    if (!authToken) {
      console.warn('⚠️ No auth token found, skipping analytics update');
      return false;
    }

    if (!deviceId) {
      console.warn('⚠️ No device ID found, skipping analytics update');
      return false;
    }

    // Call backend API to record reply
    const response = await fetch(`${backendUrl}/api/analytics/reply`, {
      method: 'POST',
      headers: await buildAuthHeaders(true),
      body: JSON.stringify({
        responseTime: responseTime,
        success: success,
        customerId: customerId  // ✅ Send customer ID
      })
    });

    if (!response.ok) {
      // Check if it's a limit reached or plan expired error (403)
      if (response.status === 403) {
        const errorData = await response.json().catch(() => ({}));
        const errorMessage = errorData.message || 'Access denied';

        console.error('❌ Access denied:', errorMessage);
        sendLogToPopup('ERROR', `⚠️ ${errorMessage}`);
        sendLogToPopup('ERROR', '🛑 Bot stopped automatically. Please upgrade your plan to continue.');

        // Stop the bot automatically
        handleStopBot();

        // Check if it's plan expired or reply limit reached
        if (errorData.planExpired) {
          // Plan has expired
          chrome.runtime.sendMessage({
            action: 'planExpired',
            data: {
              message: errorMessage,
              expireDate: errorData.expireDate
            }
          }).catch(() => { });
        } else {
          // Reply limit reached
          chrome.runtime.sendMessage({
            action: 'repliesLimitReached',
            data: {
              message: errorMessage,
              repliesThisMonth: errorData.repliesThisMonth,
              repliesLimit: errorData.repliesLimit
            }
          }).catch(() => { });
        }

        return false;
      }
      throw new Error(`Analytics API error: ${response.status}`);
    }

    const result = await response.json();
    console.log('✅ Analytics updated in MongoDB:', result.stats);

    if (result.stats) {
      const { repliesThisMonth, repliesLimit, repliesRemaining } = result.stats;

      const appStateData = await chrome.storage.local.get(['appState']);
      const appState = appStateData.appState || {};
      appState.repliesThisMonth = repliesThisMonth;
      appState.planLimits = appState.planLimits || {};
      if (repliesLimit !== undefined) {
        appState.planLimits.repliesPerMonth = repliesLimit;
      }
      await chrome.storage.local.set({ appState });

      chrome.runtime.sendMessage({
        action: 'updateRepliesUsage',
        data: { repliesThisMonth, repliesLimit, repliesRemaining }
      });
    }

    // Fetch updated analytics and notify popup
    const analyticsResponse = await fetch(`${backendUrl}/api/analytics`, {
      method: 'GET',
      headers: await buildAuthHeaders(false)
    });

    if (analyticsResponse.ok) {
      const analyticsData = await analyticsResponse.json();

      if (analyticsData.success && analyticsData.analytics) {
        // Send updated analytics to popup
        chrome.runtime.sendMessage({
          action: 'updateAnalytics',
          data: {
            totalReplies: analyticsData.analytics.totalReplies || 0,
            repliesToday: analyticsData.analytics.today.replies || 0,
            avgTime: Math.round(analyticsData.analytics.overall?.avgResponseTime || analyticsData.analytics.avgResponseTime || 0),
            successRate: Math.round(analyticsData.analytics.overall?.successRate || analyticsData.analytics.overallSuccessRate || 100),
            activeChats: analyticsData.analytics.totalUniqueCustomers || 0,  // ✅ Lifetime unique customers (Analytics tab)
            activeConversations: analyticsData.analytics.today.uniqueCustomersCount || 0  // ✅ Today's unique customers (Bot Controls)
          }
        }).catch(() => { });

        console.log('📊 Analytics synced to popup:', {
          totalReplies: analyticsData.analytics.totalReplies,
          repliesToday: analyticsData.analytics.today.replies
        });
      }
    }

    sendLogToPopup('SUCCESS', '📊 Analytics updated');
    return true;

  } catch (error) {
    console.error('❌ Analytics update error:', error);
    sendLogToPopup('WARNING', `⚠️ Analytics update failed: ${error.message}`);
    return false;
  }
}

// ✅ NEW BACKEND-FIRST: Handle full reply generation with backend service
async function handleGenerateReplyFull(customerId, newMessage, buyerUsername, gigInfo) {
  console.log('\n🚀 ===== BACKEND-FIRST REPLY GENERATION =====');
  console.log(`Customer ID: ${customerId}`);
  console.log(`Buyer Username: ${buyerUsername}`);
  // ✅ Seller name will be fetched from database by backend
  console.log(`Message Length: ${newMessage?.length} chars`);
  console.log(`Has Gig Info: ${!!gigInfo}`);

  try {
    // Get auth context
    const { token: authToken, backendUrl } = await getAuthContext();

    if (!authToken) {
      console.error('❌ No auth token found');
      return {
        success: false,
        error: 'Authentication required. Please login.'
      };
    }

    if (!backendUrl) {
      console.error('❌ No backend URL configured');
      return {
        success: false,
        error: 'Backend URL not configured'
      };
    }

    // Call NEW backend endpoint
    console.log(`📤 Calling: ${backendUrl}/api/ai/generate-reply-full`);

    const response = await fetch(`${backendUrl}/api/ai/generate-reply-full`, {
      method: 'POST',
      headers: await buildAuthHeaders(true),
      body: JSON.stringify({
        customerId,
        newMessage,
        buyerUsername,
        // ✅ sellerName is NOT sent - backend uses user.sellerName from database
        gigInfo
      })
    });

    console.log(`📥 Response status: ${response.status}`);

    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}));
      console.error('❌ Backend error:', errorData);

      // Handle specific errors
      if (response.status === 403) {
        if (errorData.planExpired) {
          return {
            success: false,
            error: 'Your subscription has expired. Please upgrade your plan.',
            planExpired: true
          };
        }
        if (errorData.repliesLimitReached) {
          return {
            success: false,
            error: errorData.message || 'Reply limit reached',
            repliesLimitReached: true,
            repliesThisMonth: errorData.repliesThisMonth,
            repliesLimit: errorData.repliesLimit
          };
        }
      }

      return {
        success: false,
        error: errorData.message || `Backend error: ${response.status}`
      };
    }

    const data = await response.json();
    console.log('✅ Backend response received');
    console.log(`Reply: "${data.reply?.substring(0, 50)}..."`);
    console.log(`Tokens used: ${data.tokensUsed}`);
    console.log(`Conversation length: ${data.conversationLength} messages`);
    console.log(`Has summary: ${data.hasSummary}`);

    if (!data.success || !data.reply) {
      console.error('❌ Invalid backend response');
      return {
        success: false,
        error: data.error || 'No reply generated'
      };
    }

    console.log('✅ ===== BACKEND-FIRST REPLY SUCCESS =====\n');

    // Return success
    return {
      success: true,
      reply: data.reply,
      tokensUsed: data.tokensUsed,
      usage: data.usage,
      conversationLength: data.conversationLength,
      hasSummary: data.hasSummary,
      repliesThisMonth: data.repliesThisMonth,
      repliesLimit: data.repliesLimit
    };

  } catch (error) {
    console.error('❌ handleGenerateReplyFull error:', error);
    return {
      success: false,
      error: error.message || 'Failed to generate reply'
    };
  }
}

console.log('Fiverr Auto Reply Bot background script loaded');

// ==========================================
// ✨ KEEP ALIVE SYSTEM (Offscreen Document)
// ==========================================

let creatingOffscreenParams; // Global promise for offscreen creation

async function setupOffscreenDocument(path) {
  // Check if offscreen document already exists
  if (chrome.runtime.getContexts) {
    const existingContexts = await chrome.runtime.getContexts({
      contextTypes: ['OFFSCREEN_DOCUMENT']
    });

    if (existingContexts.length > 0) {
      return;
    }
  } else {
    // Fallback for older Chrome versions (though MV3 usually implies newer)
    // We can't easily check, so we try to create and catch error
  }

  // Create offscreen document
  if (creatingOffscreenParams) {
    await creatingOffscreenParams;
  } else {
    try {
      creatingOffscreenParams = chrome.offscreen.createDocument({
        url: path,
        reasons: ['BLOBS'], // Using BLOBS as a valid reason that allows long running
        justification: 'Keep service worker alive for long-running bot automation'
      });
      await creatingOffscreenParams;
      creatingOffscreenParams = null;
      console.log('✅ Offscreen document created for keep-alive');
    } catch (error) {
      // Ignore if it already exists/error
      creatingOffscreenParams = null;
      console.log('Offscreen creation info:', error.message);
    }
  }
}

async function closeOffscreenDocument() {
  try {
    if (chrome.offscreen) {
      await chrome.offscreen.closeDocument();
      console.log('✅ Offscreen document closed');
    }
  } catch (error) {
    // Ignore error if no document to close
    console.log('Info closing offscreen document (might not exist):', error.message);
  }
}









